' (c) Copyright 2024, Michael Harnad
'import "pkg:/components/UILogic/VotingBookmarkLogic.bs"
'import "pkg:/components/UILogic/Watched.bs"


' Helper function convert Associative Array to Node
function ContentListToSimpleNode(contentList as Object, nodeType = "ContentNode" as String) as Object
    result = CreateObject("roSGNode", nodeType) ' create node instance based on specified nodeType
    if result <> invalid
        ' go through contentList and create node instance for each item of list
        for each arrayItem in contentList
            item = CreateObject("roSGNode", nodeType)
            item.SetFields(arrayItem)
            result.AppendChild(item)
        end for
    end if
    return result
end function

'''''''''
' SearchContentListToSimpleNode: Converts a Search list to a simple content node.
' 
' @param {object} I: _searchList - the list of search items.
' @return {object} - Returns a simple Content node.
'''''''''
function SearchContentListToSimpleNode(_searchList as object) as object
    rootcontentNode = CreateObject("roSGNode", "ContentNode")
    if rootcontentNode <> invalid
        for each searchItem in _searchList
            item = CreateObject("roSGNode", "ContentNode")
            item.addFields(searchItem)
            item.SetFields(searchItem)
            rootcontentNode.AppendChild(searchItem)
        end for
    end if
    return rootcontentNode
end function

'''''''''
' EpisodeContentListToSimpleNode: Converts an Episode List to a Simple Content Node.
' 
' @param {object} I: contentList - the Episode list.
' @param {string} [nodeType="ContentNode"]
' @return {object} - Returns the Simple Content node.
'''''''''
function EpisodeContentListToSimpleNode(contentList as Object, nodeType = "ContentNode" as String) as Object
    result = CreateObject("roSGNode", nodeType) ' create node instance based on specified nodeType
    if result <> invalid
        ' go through contentList and create node instance for each item of list
        for each arrayItem in contentList
            item = CreateObject("roSGNode", nodeType)
            item.addFields(arrayItem)
            item.SetFields(arrayItem)
            result.AppendChild(item)
        end for
    end if
    return result
end function

' Helper function converts seconds to mm:ss format.
' getTime(138) returns 2:18.
function GetTime(totalSeconds as Integer) as String
    intHours = totalSeconds \ 3600
    intMinutes = (totalSeconds MOD 3600) \ 60
    intSeconds = (totalSeconds MOD 60)
    if intHours > 0
        duration = intHours.toStr() + "h "
        if intMinutes > 0
            duration += intMinutes.toStr() + "m "
        end if
        if intSeconds > 0
            duration += intSeconds.toStr() + "s"
        end if
    else if intMinutes > 0
        duration = intMinutes.toStr() + "m "
        if intSeconds > 0
            duration += intSeconds.toStr() + "s"
        end if
    else
        duration = intSeconds.toStr() + "s"
    end if
    return duration
end function

'''''''''
' GetHHMMSSFromTotalSeconds: Converts total seconds into "HH:MM:SS".
' 
' @param {integer} I: _totalSeconds - total seconds.
' @return {string} - returns the converted "HH:MM:SS" string.
'''''''''
function GetHHMMSSFromTotalSeconds(_totalSeconds as integer) as string
    intHours = _totalSeconds \ 3600
    intMinutes = (_totalSeconds MOD 3600) \ 60
    intSeconds = (_totalSeconds MOD 60)
    return "%02d:%02d:%02d".Format(intHours, intMinutes, intSeconds)
end function

'''''''''
' CloneChildren: Clone the children of an object.
' 
' @param {object} I: node - the parent node.
' @param {integer} I: [startItem=0] - the starting index.
' @return {dynamic} - returns the clonded children.
'''''''''
function CloneChildren(node as Object, startItem = 0 as Integer)
    numOfChildren = node.GetChildCount() ' get number of row items.
    ' populate children array only wth items started from the selected one.
    ' example: row has 3 items.  user selects second one so we must take just second and third items.
    children = node.GetChildren(numOfChildren - startItem, startItem)
    childrenClone = []
    ' go through each item of children and clone them.
    for each child in children
        ' we need to clone item node because it will be damaged in case of video node content invalidation.
        childrenClone.Push(child.Clone(true))
    end for
    return childrenClone
end function

'''''''''
' GetCustomAttributes: Gets the custom attributes and assigns them to the global node.
' 
'''''''''
function GetCustomAttributes() as void
    customAttributes = {}
    ' Read the custom attributes.
    attributeNames = ParseJson(ReadAsciiFile("pkg:/source/custom_attributes.json"))
    for each attribute in attributeNames.items()
        if attribute.value.toStr() = "true"
            customAttributes[attribute.key] = true
        else if attribute.value.toStr() = "false"
            customAttributes[attribute.key] = false
        else
            customAttributes[attribute.key] = attribute.value
        end if
    end for
    ' Assign the custom attributes to the global node.
    m.global.addFields(customAttributes)
end function

'''''''''
' CreateMessageDialog: Creates a message dialog.
' 
' @param {string} title
' @param {object} message
' @param {object} buttons
' @return {object}
'''''''''
function CreateMessageDialog(title as string, message as object, buttons as object) as object
    Dialog = createObject("roSGNode", "StandardMessageDialog")
    Dialog.title = title
    Dialog.message = message
    Dialog.buttons = buttons
    return Dialog
end function

' Helper function finds child node by specified contentId
function FindNodeById(content as Object, contentId as String) as Object
    for each element in content.GetChildren(- 1, 0)
        if element.id = contentId
            return element.clone(true)
        else if element.getChildCount() > 0
            result = FindNodeById(element, contentId)
            if result <> invalid
                return result.clone(true)
            end if
        end if
    end for
    return invalid
end function

'''''''''
' DebugListRegistry: Used to list the contents of the registry for debugging.
'                    DEBUG ONLY!
' 
' @return {dynamic}
'''''''''
function DebugListRegistry()
    registry = CreateObject("roRegistry")
    sections = registry.GetSectionList()
    if sections.Count() > 0
        for each section in sections
            ? ">> DEBUG: Registry section: " + section
            iRegistrySection = CreateObject("roRegistrySection", section)
            keyList = iRegistrySection.GetKeyList()
            for each key in keyList
                ? ">>> DEBUG: key is " + key
                ? ">>>> DEBUG: value is " + iRegistrySection.Read(key)
            end for
        end for
    else
        ? ">> DEBUG:  No registry sections available."
    end if
end function

'''''''''
' DebugClearRegistry: Used to clear the registyr entries.
'                     DEBUG ONLY!
' 
' @return {dynamic}
'''''''''
function DebugClearRegistry()
    registry = CreateObject("roRegistry")
    sections = registry.GetSectionList()
    if sections <> invalid
        for each section in sections
            ? ">> DEBUG: Removed registry section - " + section
            registry.Delete(section)
        end for
    end if
end function

'''''''''
' GetFeedItemFromContent: Gets a feed item from the content object.
' 
' @param {object} I: content - the content object.
' @param {integer} I: index - index in the content object.
' @return {object} - returns the feed item object.
'''''''''
function GetFeedItemFromContent(content as object, index as integer) as object
    if content <> invalid
        feedItem = content.getChild(index)
    end if
    return feedItem
end function

'''''''''
' GetAllSeasonsFromSeries: Gets all Seasons from a Series.
' 
' @param {object} I: _series - the Series object.
' @return {object} - returns an array of Seasons.
'''''''''
function GetAllSeasonsFromSeries(_series as object) as object
    ' Define a Seasons array.
    seasons = []
    if _series.hasSeasons = true
        seasons = _series.getChildren(- 1, 0)
    end if
    return seasons
end function

'''''''''
' GetSeasonFromSeries: Gets the Season from a Series.
' 
' @param {object} I: _series - the Series object.
' @param {integer} I: _index - the Season index.
' @return {object} - returns the Season object if valid, else, invalid.
'''''''''
function GetSeasonFromSeries(_series as object, _index as integer) as object
    if _series.hasSeasons = true
        return _series.getChild(_index)
    else
        return invalid
    end if
end function

'''''''''
' GetEpisodeFromSeriesSeasonById: Gets an Episode from a Series that contains Seasons based on the Season
'  title and Episode ID.
' 
' @param {object} I: _series - the Series feed object.
' @param {string} I: _episodeId - Episode ID.
' @param {string} I: _season - the Season title.
' @return {object} - returns the Episode object if found, else, returns invalid.
'''''''''
function GetEpisodeFromSeriesSeasonById(_series as object, _episodeId as string, _season as string) as object
    seasons = _series.getChildren(- 1, 0)
    if seasons <> invalid
        for each season in seasons
            if season.title = _season
                episodes = season.getChildren(- 1, 0)
                for each episode in episodes
                    if episode.id = _episodeId
                        return episode
                    end if
                end for
            end if
        end for
    end if
    return invalid
end function

'''''''''
' GetEpisodeFromSeriesById: Gets an Episode from a Series based on the Episode ID.
' 
' @param {object} I: _series - the Series feed object.
' @param {string} I: _episodeId - Episode ID.
' @return {object} - returns the Episode object if found, else, returns invalid.
'''''''''
function GetEpisodeFromSeriesById(_series as object, _episodeId as string) as object
    episodes = _series.getChildren(- 1, 0)
    if episodes <> invalid
        for each episode in episodes
            if episode.id = _episodeId
                return episode
            end if
        end for
    end if
    return invalid
end function

'''''''''
' GetEpisodeFromSeries: Gets an Episode from a Series based on the Episode index.
' 
' @param {object} I: _series - the Series feed object.
' @param {integer} I: _index - index into the Episodes array.
' @return {object} - returns the requested Episode.
'''''''''
function GetEpisodeFromSeries(_series as object, _index as integer) as object
    if _series <> invalid
        episode = _series.getChild(_index)
    end if
    return episode
end function

'''''''''
' GetEpisodeFromSeason: Gets an Episode from a Season based on the Episode index.
' 
' @param {object} I: season - the Season feed object.
' @param {integer} I: index - index into the Episodes array.
' @return {object} - returns the requested Episode.
'''''''''
function GetEpisodeFromSeason(season as object, index as integer) as object
    if season <> invalid
        episode = season.getChild(index)
    end if
    return episode
end function

'''''''''
' GetEpisodeIndex: Get the Episode index within a Series.
' 
' @param {object} I: _episode - the episode object.
' @param {object} O: _seasonIndex - the Season index if the Episode is part of a Season.
' @return {integer} - returns the Episode index.
'''''''''
function GetEpisodeIndex(_episode as object, _seasonIndex as object) as integer
    seriesParent = GetSeriesParentObjectFromEpisode(_episode)
    if seriesParent.seasons <> invalid
        seasonIndex = 0
        for each season in seriesParent.seasons
            episodes = season.episodes
            ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
            episodes.SortBy("releaseDate", "r")
            ' Reindex the Episodes since they have been sorted.
            ReindexEpisodes(episodes)
            for each episode in episodes
                if episode.id = _episode.id
                    _seasonIndex[0] = seasonIndex.toStr()
                    return episode.episodeIndex
                end if
            end for
            seasonIndex++
        end for
    else
        episodes = seriesParent.episodes
        ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
        episodes.SortBy("releaseDate", "r")
        ' Reindex the Episodes since they have been sorted.
        ReindexEpisodes(episodes)
        for each episode in episodes
            if episode.id = _episode.id
                return episode.episodeIndex
            end if
        end for
    end if
end function

'''''''''
' ReindexEpisodes: Reindex the Episodes.
' 
' @param {object} I: _episodes - the episode array.
' 
'''''''''
function ReindexEpisodes(_episodes as object) as void
    episodeIndex = 0
    for each episode in _episodes
        episode.episodeIndex = episodeIndex
        episodeIndex++
    end for
end function

'''''''''
' GetAllEpisodesFromSeries: Get all Epsiodes for a Series.
' 
' @param {object} I: _series - the Series object.
' @return {object} - returns an array of Episodes.
'''''''''
function GetAllEpisodesFromSeries(_series as object) as object
    requestedEpisodes = []
    if _series.hasSeasons <> true
        episodes = CloneChildren(_series)
        for each episode in episodes
            requestedEpisodes.Push(episode)
        end for
    else
        seasons = _series.GetChildren(- 1, 0)
        for each season in seasons
            episodes = CloneChildren(season)
            for each episode in episodes
                requestedEpisodes.Push(episode)
            end for
        end for
    end if
    return requestedEpisodes
end function

'''''''''
' GetAllEpisodesFromSeason: Get all Episodes for a Season.
' 
' @param {object} I: _season - the Season object.
' @return {object} - returns an array of Episodes.
'''''''''
function GetAllEpisodesFromSeason(_season as object) as object
    requestedEpisodes = []
    if _season.getChildCount() > 0
        episodes = CloneChildren(_season)
        for each episode in episodes
            requestedEpisodes.Push(episode)
        end for
    end if
    return requestedEpisodes
end function

'''''''''
' GetAllEpisodesStartingAt: Get all Episodes from a Series starting at an Episode index.
' 
' @param {object} I: _series - the Series object.
' @param {integer} I: _startIndex - the starting Episode index.
' @return {object} - returns an array of Episodes.
'''''''''
function GetAllEpisodesStartingAt(_series as object, _startIndex as integer) as object
    episodes = []
    requestedEpisodes = []
    ' If the "episodes" array is invalid, use the 'children' array.
    if _series.episodes = invalid
        ' Get all Series Episodes.
        seriesEpisodes = _series.getChildren(- 1, 0)
        ' Make a copy of each Episode.
        for each child in seriesEpisodes
            episodes.Push(child.Clone(true))
        end for
    else
        ' Build a 'children' array of Episodes.
        seriesEpisodes = _series.episodes
        for each episode in seriesEpisodes
            ' Create a ContentNode for each Episode.
            episodeNode = CreateObject("roSGnode", "ContentNode")
            episodeNode.addFields(episode)
            episodeNode.addFields({
                "url": episode.content.videos[0].url
            })
            episodes.Push(episodeNode)
        end for
    end if
    ' If requested Episode index is 0, return the entire array.
    if _startIndex = 0
        requestedEpisodes.Append(episodes)
    else
        ' Filter the Episode array starting at the requested index.
        for index = _startIndex to episodes.Count() - 1 step 1
            requestedEpisodes.Push(episodes[index])
        end for
    end if
    return requestedEpisodes
end function

'''''''''
' ContentIsType: Checks the media type of a content object.
' 
' @param {object} I: _feedContent the feed content to check.
' @param {string} I: _type - the type to check.
' @return {boolean} - returns true if content is 'type', else, false.
'''''''''
function ContentIsType(_feedContent as object, _type as string) as boolean
    return bslib_ternary(_feedContent.mediaType = _type, true, false)
end function

'''''''''
' SeriesHasSeasons: Checks to see if a Series has Seasons.
' 
' @param {object} I: _series - the Series object.
' @return {boolean} - returns trus if Series has Seasons, else, false.
'''''''''
function SeriesHasSeasons(_series as object) as boolean
    return _series.hasSeasons
end function

'''''''''
' SetPlayAllFlag: Sets the PlayAll flag on the feed content object.
' 
' @param {object} I: feedContent - the feed content.
' @param {boolean} I: bPlayall - true, or, false.
' @return {object} - returns the feed content.
'''''''''
function SetPlayAllFlag(feedContent as object, bPlayall as boolean) as object
    feedContent.removeFields([
        "PlayAll"
    ])
    feedContent.addFields({
        "PlayAll": bPlayall
    })
    return feedContent
end function

'''''''''
' ShowADialog: Shows a dialog on the screen.
' 
' @param {string} I: title - title of the dialog.
' @param {string} I: message - message to display.
' @param {object} I: buttons - buttons to display.
' @return {object} returns the dialog object.
'''''''''
function ShowAMessageDialog(_title as string, _message as object, _buttons as object, _width = 0.0 as float, _height = 0.0 as float, _opacity = 1.0 as float) as object
    ' Create the dialog.
    modalDialog = CreateMessageDialog(_title, _message, _buttons)
    modalDialog.width = _width
    modalDialog.height = _height
    modalDialog.opacity = _opacity
    modalDialog.translation = [
        "25, 25"
    ]
    modalDialog.palette = CreateADialogPalette()
    scene = m.top.getScene()
    scene.dialog = modalDialog
    return modalDialog
end function

'''''''''
' ConvertAdbreakToSeconds: Converts Adbreak time to total seconds.
' 
' @param {string} I: adbreak - the adbreak string.
' @return {integer} - returns the adbreak in total seconds.
'''''''''
function ConvertAdbreakToSeconds(adbreak as string) as integer
    stringObj = CreateObject("roString")
    stringObj.SetString(adbreak)
    ' Make sure adbreak is 'mm:ss' format.
    if stringObj.Instr(":") = - 1
        totalSeconds = stringObj.toInt()
    else
        time = stringObj.Split(":")
        hours = time[0].toInt()
        minutes = time[1].toInt()
        seconds = time[2].toInt()
        totalSeconds = (hours * 60) * 60
        totalSeconds += minutes * 60
        totalSeconds += seconds
    end if
    return totalSeconds
end function

'''''''''
' GenerateGuid: Generates a GUID.
' 
' @return {string} - returns a GUID.
'''''''''
function GenerateGuid() as string
    thisDevice = CreateObject("roDeviceInfo")
    return thisDevice.GetRandomUUID()
end function

'''''''''
' GetChannelGuid: Gets the unique channel ID.
' 
' @return {string} - returns the ID.
'''''''''
function GetChannelGuid() as string
    thisDevice = CreateObject("roDeviceInfo")
    return thisDevice.GetChannelClientId()
end function

'''''''''
' GenerateARandomPassword: Generates a random password.
' 
' @return {string} - returns a random password.
'''''''''
function GenerateARandomPassword() as string
    randomPassword = ""
    Chars = [
        "A"
        "@"
        "B"
        "#"
        "C"
        "$"
        "D"
        "%"
        "E"
        "!"
        "F"
        "&"
        "*"
        "a"
        "b"
        "c"
        "d"
        "e"
        "f"
        "z"
    ]
    while Len(randomPassword) < 20
        randomInt = RND(19)
        randomPassword += Chars[randomInt]
    end while
    return randomPassword
end function

'''''''''
' CreateADialogPalette: Creates a palette object.
' 
' @return {object} - returns a palette object.
'''''''''
function CreateADialogPalette() as object
    ' Create a Palette for the dialog.
    palette = CreateObject("roSGNode", "RSGPalette")
    palette.colors = {
        DialogBackgroundColor: "0xffffff" ' white
        DialogTextColor: "0x000000" ' black
        DialogSecondaryTextColor: "0x000000" ' black
        DialogFocusColor: "0x7f00ff" ' violet
        DialogFocusItemColor: "0xffffff" ' white
    }
    return palette
end function

'''''''''
' TrimAString: Trims a string of leading and trailing spaces.
' 
' @param {string} I: _string - the string to trim.
' @return {string} - returns the trimmed string.
'''''''''
function TrimAString(_string as string) as string
    ' Create a String object and init it with the passed string.
    stringObj = CreateObject("roString")
    stringObj.SetString(_string)
    return stringObj.Trim()
end function

'''''''''
' UpdateLoadingIndicator: Updates the text of the loading indicator.
' 
' @param {string} I: _text - the updated text.
' @return {dynamic}
'''''''''
function UpdateLoadingIndicator(_text as string)
    scene = m.top.getScene()
    loadingIndicator = scene.FindNode("loadingIndicator")
    if loadingIndicator <> invalid
        loadingIndicator.text = _text
    end if
end function

'''''''''
' GetTimeRemaining: Get the time remaining for content that was stopped.
' 
' @param {float} I: _duration - the total content duration in seconds.
' @param {object} I: _position - the current content position in seconds.
' @return {string}
'''''''''
function GetTimeRemaining(_duration as float, _position as object) as string
    remaining = ""
    ' Get the total seconds remaining.
    totalseconds = _duration - _position[0]
    ' Get the seconds part.
    seconds = totalseconds mod 60
    ' Get the minutes part.
    minutes = fix(totalseconds / 60)
    ' If minutes are greater than zero...
    if minutes > 0
        remaining = minutes.toStr() + "m"
        if seconds > 0
            remaining += ", " + seconds.toStr() + "s remaining"
        end if
    else
        if seconds > 0
            remaining = seconds.toStr() + "s remaining"
        end if
    end if
    return remaining
end function

'''''''''
' VoteHasBeenCast: Checks to see if a vote has already been cast.
' 
' @param {string} I: _contentID - the content ID.
' @param {object} O: _castvote - the cast vote if it has already been cast.
' @return {boolean} - returns true if a vote was cast for this content ID, else, false.
'''''''''
function VoteHasBeenCast(_contentID as string, _castvote as object) as boolean
    voting = VotingBookmarkClass(_contentID, "")
    cast = voting.ReadBookmark(_contentID)
    _castvote[0] = cast
    return bslib_ternary(cast <> "", true, false)
end function

'''''''''
' GetSCreenNodeFromMain: Gets a child screen node of the Main scene.
' 
' @param {object} I: _main - the Main Sceene node.
' @param {string} I: _screenNode - the name of the screen node to get.
' @return {object} - returns the screen node if found, else, invalid.
'''''''''
function GetSCreenNodeFromMain(_main as object, _screenNode as string) as object
    childScreen = invalid
    ' Find the ScreenNode.
    sceneChildren = _main.getChildren(- 1, 0)
    for each child in sceneChildren
        if child.subtype() = _screenNode
            childScreen = child
        end if
    end for
    return childScreen
end function

'''''''''
' DebugPrint: Prints a debug message to the console.
' 
' @param {string} I: _message - message to print.
'''''''''
function DebugPrint(_message as string) as void
    if m.global.DEBUG_PRINT <> invalid and m.global.DEBUG_PRINT
        Print "[DEBUG] " + _message
    end if
end function

'''''''''
' DebugPrintObject: Prints a debug message for an object.
' 
' @param {object} I: _object - the object.
'''''''''
function DebugPrintObject(_object as object) as void
    if m.global.DEBUG_PRINT <> invalid and m.global.DEBUG_PRINT
        Print _object
    end if
end function

'''''''''
' DebugPrintAssocArray: Prints a debug message showing the contents of an associative array.
' 
' @param {object} I: _array = the associative array.
'''''''''
function DebugPrintAssocArray(_array as object) as void
    for each item in _array.items()
        if Type(item.value) = "Object"
            DebugPrintAssocArray(item.value)
        end if
        debugString = CreateObject("roString")
        if Type(item.value) = "String"
            debugString.SetString("%s %s")
        else
            debugString.SetString("%s %d")
        end if
        dbgString = debugString.Format(item.key, item.value)
        DebugPrint(dbgString)
    end for
end function

'''''''''
' StringContains: Check for a substring within a string.
' 
' @param {string} I: _str - the string to check.
' @param {string} I: _substr - the substring.
' @return {boolean} - returns true if found, else, false.
'''''''''
function StringContains(_str as string, _substr as string) as boolean
    stringObj = CreateObject("roString")
    stringObj.SetString(LCase(_str))
    index = stringObj.Instr(LCase(_substr))
    return bslib_ternary(index > - 1, true, false)
end function

'''''''''
' ExtractRawContentFromFeed: Extracts a content feed item from the feed.
' 
' @param {string} I: _mediaType - the feed type.
' @param {string} I: _id - the feed item id.
' @param {object} I: [_feedFile=invalid] - the feed file object.
' @return {object} - returns the requested feed item if found, else, invalid.
'''''''''
function ExtractRawContentFromFeed(_mediaType as string, _id as string, _feedFile = invalid as object) as object
    ' Massage the _mediaType if required so that the parse can find it.
    if _mediaType = "movie" then
        _mediaType = "movies"
    end if
    if _mediaType = "liveFeed" then
        _mediaType = "liveFeeds"
    end if
    if _mediaType = "shortFormVideo" then
        _mediaType = "shortFormVideos"
    end if
    if _mediaType = "tvSpecial" then
        _mediaType = "tvSpecials"
    end if
    if _feedFile = invalid
        ' Parse the feed file structure from the 'tmp' storage.
        contentFeed = ParseJson(ReadAsciiFile("tmp:/channelFeed.json"))
    else
        contentFeed = _feedFile
    end if
    ' If it's NOT an "episode", extract it from the feed file.
    if _mediaType <> "episode"
        for each feedItem in contentFeed[_mediaType]
            if feedItem.id = _id
                return feedItem
            end if
        end for
    else
        ' Drill down until we find it.
        for each series in contentFeed["series"]
            if series.seasons = invalid
                for each episode in series.episodes
                    if episode.id = _id
                        return episode
                    end if
                end for
            else
                for each season in series.seasons
                    for each episode in season.episodes
                        if episode.id = _id
                            return episode
                        end if
                    end for
                end for
            end if
        end for
    end if
    return invalid
end function

'''''''''
' CreateContentNodeFromRawObject: Create a ContentNode from raw a raw json object.
' 
' @param {object} I: _args - an associative array that identifies the object to create.
' @return {object} - returns a ContentNode built from the raw json object.
'''''''''
function CreateContentNodeFromRawObject(_args as object) as object
    ' Create a ContentNode.
    playableItem = CreateObject("roSGNode", "ContentNode")
    ' Get the raw content from the feed file.
    requestedContent = ExtractRawContentFromFeed(_args["dl_mediaType"], _args["dl_id"])
    ' If the content is a Series, get the requested Episode if provided.
    if _args["dl_mediaType"] = "series"
        ' Is the user requesting a specific Season and Episode?
        if _args.DoesExist("dl_season") = true
            ' Get the requested Season.
            requestedSeason = requestedContent.seasons[_args["dl_season"].toInt() - 1]
            ' Get the requested Episode.
            requestedEpisode = requestedSeason.episodes[_args["dl_episode"].toInt() - 1]
            playableItem.Append(requestedEpisode)
            playableItem.id = requestedEpisode.id
            playableItem.HDPosterUrl = requestedEpisode.thumbnail
            playableItem.url = requestedEpisode.content.videos[0].url
            playableItem.addFields({
                "mediaType": "episode"
                "Length": requestedEpisode.content.duration
            })
            playableItem.addFields({
                "EpisodePosition": requestedEpisode.episodeNumber.toStr()
                "hasSeasons": true
            })
            ' Is the user requesting an Episode from a Series?
        else if _args.DoesExist("dl_episode") = true
            requestedEpisode = requestedContent.episodes[_args["dl_episode"].toInt() - 1]
            playableItem.Append(requestedEpisode)
            playableItem.id = requestedEpisode.id
            playableItem.HDPosterUrl = requestedEpisode.thumbnail
            playableItem.url = requestedEpisode.content.videos[0].url
            playableItem.addFields({
                "mediaType": "episode"
                "Length": requestedEpisode.content.duration
            })
            playableItem.addFields({
                "EpisodePosition": requestedEpisode.episodeNumber.toStr()
            })
            ' User is requesting the entire Series.
        else
            ' Build parent Series.
            playableItem.Append(requestedContent)
            playableItem.id = requestedContent.id
            playableItem.HDPosterUrl = requestedContent.thumbnail
            playableItem.addFields({
                "mediaType": "series"
            })
            if requestedContent.DoesExist("seasons") = true
                playableItem.addFields({
                    "hasSeasons": true
                })
                ' Define an array to hold updated Episodes within the Season.
                seasons = []
                ' Series contains Seasons of Episodes.
                for each season in playableItem.seasons
                    for each episode in season.episodes
                        ' Add parentSeasonindex to each Episode.
                        episode.parentSeasonindex = (season.seasonNumber - 1).toStr()
                    end for
                    ' Save the updated season.
                    seasons.Push(season)
                end for
                ' Update the Seasons with the added Episode field.
                playableItem.seasons = seasons
            else
                ' Series contains Episodes only.
                playableItem.addFields({
                    "hasSeasons": false
                })
            end if
        end if
        ' User is requesting a non-Series item.
    else
        playableItem.Append(requestedContent)
        playableItem.id = requestedContent.id
        playableItem.HDPosterUrl = requestedContent.thumbnail
        playableItem.url = requestedContent.content.videos[0].url
        playableItem.length = requestedContent.content.duration
        playableItem.addFields({
            "mediaType": _args["dl_mediaType"]
        })
    end if
    return playableItem
end function

'''''''''
' GetSeriesParentObjectFromEpisode: Get the parent Series object from a child Episode.
' 
' @param {object} I: _episode - the child Episode.
' @return {object} - returns the parent Series object if found, else, invalid.
'''''''''
function GetSeriesParentObjectFromEpisode(_episode as object) as object
    seriesId = GetSeriesParentIdFromEpisode(_episode)
    series = ExtractRawContentFromFeed("series", seriesId)
    series.hasSeasons = bslib_ternary(series.seasons = invalid, false, true)
    return series
end function

'''''''''
' GetSeriesParentIdFromEpisode: Gets the Series parent ID from an Episode.
' 
' @param {object} I: _episode - the Episode content.
' @return {object} - returns the Series parent id if found, else, invalid.
'''''''''
function GetSeriesParentIdFromEpisode(_episode as object) as object
    ' Parse the feed file structure from the 'tmp' storage.
    contentFeed = ParseJson(ReadAsciiFile("tmp:/channelFeed.json"))
    ' Iterate through all Series.
    for each series in contentFeed["series"]
        ' Does the Series have Seasons?
        if series.seasons <> invalid AND series.seasons.Count() > 0
            for each season in series.seasons
                for each episode in season.episodes
                    if episode.id = _episode.id
                        return series.id
                    end if
                end for
            end for
        else
            for each episode in series.episodes
                if episode.id = _episode.id
                    return series.id
                end if
            end for
        end if
    end for
    return invalid
end function

'''''''''
' CreatePlaylistFromEpisodes: Create a playlist of all Series Episodes from a single Episode.
' 
' @param {object} I: _episode - the starting Episode object.
' @param {integer} I: _episodeIndex - the index of the starting Episode.
' @return {object} - returns a playlist that consists of all Episodes in the Series.
'''''''''
function CreatePlaylistFromEpisodes(_episode as object, _episodeIndex as integer) as object
    ' Get the Series parent from raw Json. 
    rawseriesParent = GetSeriesParentObjectFromEpisode(_episode)
    ' Create a playlist of subsequent Episodes.
    playlistNode = CreateObject("roSGNode", "ContentNode")
    ' Get all Series Episodes.
    if rawseriesParent.seasons <> invalid
        ' Seasons with Episodes.
        for each season in rawseriesParent.seasons
            ' Initialize an Episode index.
            epIndex = 0
            if season.seasonNumber >= _episode.parentSeasonindex.toInt()
                episodes = season.episodes
                ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
                episodes.SortBy("releaseDate", "r")
                for each episode in episodes
                    if epIndex >= _episodeIndex
                        childNode = CreateObject("roSGNode", "ContentNode")
                        childNode.addFields(episode)
                        childNode.id = episode.id
                        childNode.HDPosterUrl = episode.thumbnail
                        childNode.url = episode.content.videos[0].url
                        childNode.addFields({
                            "playStart": 0
                        })
                        childNode.addFields({
                            "mediaType": "episode"
                            "Length": episode.content.duration
                        })
                        childNode.addFields({
                            "parentSeasonIndex": season.seasonNumber.toStr()
                            "episodeIndex": epIndex
                        })
                        childNode.addFields({
                            "ratingObj": episode.rating
                        })
                        childNode.removeFields([
                            "rating"
                        ])
                        playlistNode.AppendChild(childNode)
                    end if
                    epIndex++
                end for
            end if
        end for
    else
        ' Episodes only.
        ' Initialize an Episode index.
        epIndex = 0
        episodes = rawseriesParent.episodes
        ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
        episodes.SortBy("releaseDate", "r")
        for each episode in episodes
            if epIndex >= _episodeIndex
                childNode = CreateObject("roSGNode", "ContentNode")
                childNode.addFields(episode)
                childNode.id = episode.id
                childNode.HDPosterUrl = episode.thumbnail
                childNode.url = episode.content.videos[0].url
                childNode.addFields({
                    "playStart": 0
                })
                childNode.addFields({
                    "mediaType": "episode"
                    "Length": episode.content.duration
                    "episodeIndex": epIndex
                })
                childNode.addFields({
                    "ratingObj": episode.rating
                })
                childNode.removeFields([
                    "rating"
                ])
                playlistNode.AppendChild(childNode)
            end if
            epIndex++
        end for
    end if
    return playlistNode
end function

'''''''''
' CreatePlaylistFromSeries: Creates a playlist of Episodes from a Series.
' 
' @param {object} I: _series - the Series object.
' @return {object} - returns an Episode playlist.
'''''''''
function CreatePlaylistFromSeries(_series as object) as object
    ' Create a content node for the starting Episode.
    episodeNode = CreateObject("roSGNode", "ContentNode")
    if _series.hasSeasons = false
        ' Series with Episodes only.
        if _series.DoesExist("childEpisodes")
            episodes = _series.childEpisodes
        else
            episodes = _series.episodes
        end if
        episodeNode.id = episodes[0].id
        episodeNode.addFields(episodes[0])
    else
        ' Series with Seasons and Episodes.
        if _series.DoesExist("seasons")
            seasons = _series.seasons
        else
            seasons = _series.getChildren(- 1, 0)
        end if
        ' Get the Episodes for the first Season.
        if seasons[0].DoesExist("childEpisodes")
            episodes = seasons[0].childEpisodes
        else
            episodes = seasons[0].episodes
        end if
        episodeNode.id = episodes[0].id
        episodeNode.addFields(episodes[0])
    end if
    return CreatePlaylistFromEpisodes(episodeNode, 0)
end function

'''''''''
' FileExists: Checks for the existence of a file.
' 
' @param {string} I: _directory - the root directory.
' @param {string} I: _file - the file to check for.
' @return {boolean} - returns true if found, else, false.
'''''''''
function FileExists(_directory as string, _file as string) as boolean
    result = false
    fileList = ListDir(_directory)
    for each file in fileList
        if file = _file
            result = true
            exit for
        end if
    end for
    return result
end function

'''''''''
' MarkVideoAsWatched: Create a 'Watched' bookmark for a video.
' 
' @param {string} I: _contentId - the content ID.
'''''''''
function MarkVideoAsWatched(_contentId as string) as void
    ' Create a 'Watched' bookmark.
    watchedBm = Watched()
    today = CreateObject("roDateTime")
    watchedBm.WriteBookmark(_contentId, today.AsDateString("long-date"))
end function

'''''''''
' VideoHasBeenWatched: Checks to see if a video has been previously watched.
' 
' @param {string} I: _contentId - the content id.
' @return {boolean} - returns true if video has been watched, else, false.
'''''''''
function VideoHasBeenWatched(_contentId as string) as boolean
    beenWatched = false
    ' Get a Watched bookmark.
    watchedBM = Bookmark("Watched")
    ' Get the Watched keys.
    watchedKeyList = watchedBM.GetKeyList()
    ' If the Watched keylist contains items, scan them...
    if watchedKeyList.Count() > 0
        for each key in watchedKeyList
            if key = _contentId
                beenWatched = true
                exit for
            end if
        end for
    end if
    return beenWatched
end function

'''''''''
' AppendEpisodesToSeriesNode: Appends a list of Episodes to the Series parent node.
' 
' @param {object} I: _series - the Series node object.
' @param {boolean} I: _hasSeasons - flag that indicates if the Series has Seasons.
' @return {object} - returns the update Series node object.
'''''''''
function AppendEpisodesToSeriesNode(_series as object, _hasSeasons as boolean) as object
    updatedSeries = _series
    if _hasSeasons = false
        ' Create an Episode child node to append to the Series.
        for each episode in updatedSeries.childEpisodes
            episodeNode = CreateObject("roSGNode", "ContentNode")
            episodeNode.addFields(episode)
            episodeNode.SetFields({
                "id": episode.id
            })
            updatedSeries.appendChild(episodeNode)
        end for
    else
        ' Get the Series child Seasons.
        seasons = updatedSeries.getChildren(- 1, 0)
        for each season in seasons
            for each episode in season.childEpisodes
                ' Create an Episode child node to append to the Series.
                episodeNode = CreateObject("roSGNode", "ContentNode")
                episodeNode.addFields(episode)
                episodeNode.SetFields({
                    "id": episode.id
                })
                season.appendChild(episodeNode)
            end for
        end for
    end if
    return updatedSeries
end function

'''''''''
' FindNextEpisodeInPlaylist: Find the next Episode in a playlist.
' 
' @param {object} I: _currentEpisode - the current Episode.
' @param {object} I: _playlist - the playlist.
' @return {object} - returns the next Episode.
'''''''''
function FindNextEpisodeInPlaylist(_currentEpisode as object, _playlist as object) as object
    ' Get the list of Episodes in the playlist.
    episodes = _playlist.getChildren(- 1, 0)
    nextEpisode = invalid
    playlistIndex = 0
    for each episode in episodes
        if _currentEpisode.title = episode.title
            nextEpisode = _playlist.getChild(playlistIndex + 1)
            exit for
        end if
        playlistIndex++
    end for
    return nextEpisode
end function

'''''''''
' GetRokuDisplaySize: Gets the size of the attached Roku device.
' 
' @return {object} - returns the size as a associative array.
'''''''''
function GetRokuDisplaySize() as object
    ' Get a reference to the Device Info object.
    deviceInfo = CreateObject("roDeviceInfo")
    return deviceInfo.GetDisplaySize()
end function

'''''''''
' IsAudioContent: Checks for an Audio-only object.
' 
' @param {string} I: _targetContent - the target content object.
' @return {boolean} - returns true if url is .mp3, or, false.
'''''''''
function IsAudioContent(_targetContent as object) as boolean
    return bslib_ternary(_targetContent.mediaType = "audioOnly", true, false)
end function'//# sourceMappingURL=./utils.bs.map