' (c) Copyright 2024, Michael Harnad
'import "pkg:/source/utils.bs"
'import "pkg:/source/Theme.bs"


sub Init()
    m.top.functionName = "SearchForContent"
    ' Set the theme for the Search Screen.
    m.theme = Theme("SearchScreen")
end sub

'''''''''
' SearchForContent: Searches through the search indices for a match on search input.
' 
'''''''''
function SearchForContent() as void
    ' Get the parent Search screen.
    m.searchScreen = m.top.getParent()
    ' Get a reference to the Search results list.
    m.searchList = m.searchScreen.FindNode("searchList")
    ' Create the parent node for the Search results.
    m.searchList.content = CreateObject("roSGNode", "ContentNode")
    ' Get the search results label.
    m.searchResultsLabel = m.top.searchScreen.FindNode("searchResults")
    ' Set the font for the label.
    m.theme.SetFont(m.searchResultsLabel)
    ' Define a String object for use later.
    m.stringObject = CreateObject("roString")
    ' Define an array to hold the results.
    searchContent = []
    ' Define an array that will define the search set.
    searchIndex = []
    ' Define a temporary array to hold intermediate results.
    searchResults = []
    ' Load prior search results to improve performance.     
    priorSearchResults = LoadSearchResults(m.top.textInput)
    ' If we have a prior search subset, use it.  Otherwise, use the global content index.
    if priorSearchResults <> invalid AND priorSearchResults.Count() > 0
        searchIndex = priorSearchResults
    else
        searchIndex = m.top.searchIndex
    end if
    ' Define some arrays to hold the totals of each type of content.
    foundMovies = []
    foundLiveFeeds = []
    foundSeries = []
    foundEpisodes = []
    foundShortFormVideos = []
    foundTvSpecials = []
    ' Scan the Search index looking for a match.
    for each indexEntry in searchIndex
        foundContent = false
        ' Search within Title.
        m.stringObject.SetString(LCase(indexEntry.title))
        if m.stringObject.Instr(LCase(m.top.textInput)) <> - 1
            foundContent = true
        else
            ' Search within Long Description.
            m.stringObject.SetString(LCase(indexEntry.longDescription))
            if m.stringObject.Instr(LCase(m.top.textInput)) <> - 1
                foundContent = true
            end if
        end if
        ' If content was found, save it as Search content.
        if foundContent = true
            ' Add to the Search target for the next set of keystrokes.
            searchResults.Push(indexEntry)
            ' Extract the matching item and add it to the search list array.
            searchListItem = m.top.searchScreen.callfunc("GetContentById", indexEntry.id)
            if searchListItem <> invalid
                searchContent.Push(searchListItem)
                if indexEntry.mediatype = "movies"
                    foundMovies.Push(searchListItem)
                else if indexEntry.mediaType = "livefeeds"
                    foundLiveFeeds.Push(searchListItem)
                else if indexEntry.mediatype = "series"
                    foundSeries.Push(searchListItem)
                else if indexEntry.mediatype = "episode"
                    searchListItem.SetFields({
                        "episodeIndex": indexEntry.episodeIndex.toStr()
                    })
                    foundEpisodes.Push(searchListItem)
                else if indexEntry.mediatype = "shortFormVideos"
                    foundShortFormVideos.Push(searchListItem)
                else if indexEntry.mediatype = "tvSpecials"
                    foundTvSpecials.Push(searchListItem)
                end if
                m.searchList.content.appendChild(searchListItem)
            end if
        end if
    end for
    ' Save the separate Search content items.
    m.top.foundMovies = foundMovies
    m.top.foundLiveFeeds = foundLiveFeeds
    m.top.foundSeries = foundSeries
    m.top.foundEpisodes = foundEpisodes
    m.top.foundShortFormVideos = foundShortFormVideos
    m.top.foundTvSpecials = foundTvSpecials
    ' Save the most recent Search results.
    SaveSearchResults(searchResults)
    ' Save the Search content in our object.
    m.top.searchContent = searchContent
end function

'''''''''
' SaveSearchResults: Saves the Search results to tmp so that they can be used
'                       in subsequent searches.
' 
' @param {object} I: _results - the current search results.
'''''''''
function SaveSearchResults(_results as object) as void
    aaSearchResults = {}
    aaSearchResults["input"] = m.top.textInput
    aaSearchResults["targets"] = _results
    ' Save the Search results to tmp memory.
    tempFile = "tmp:/SearchResults" + "_" + m.top.textInput + ".json"
    WriteAsciiFile(tempFile, FormatJson(aaSearchResults))
end function

'''''''''
' LoadSearchResults: Loads the prior Search results to improve Search performance.
' 
' @param {string} I: _input - the current input string.
' @return {object} - Returns the prior Search results.
'''''''''
function LoadSearchResults(_input as string) as object
    ' Get the length of the input string.
    inputLength = Len(_input)
    if inputLength = 1
        return invalid
    end if
    ' If a matching prior Search results file does not exist, return.
    if FileExists("tmp:/", "SearchResults_" + Left(_input, inputLength - 1) + ".json") = false
        return invalid
    end if
    tempFile = "tmp:/SearchResults" + "_" + Left(_input, inputLength - 1) + ".json"
    aaSearchResults = ParseJson(ReadAsciiFile(tempFile))
    if aaSearchResults <> invalid
        return aaSearchResults["targets"]
    else
        return invalid
    endif
end function'//# sourceMappingURL=./SearchTask.bs.map