' (c) Copyright 2023, Michael Harnad, All rights reserved.
Library "Roku_Ads.brs"
'import "pkg:/components/UILogic/ScreenStackLogic.bs"
'import "pkg:/source/utils.bs"
'import "pkg:/components/UILogic/VideoPlayerWithAds.bs"
' Number of seconds before the end of the Episode to start the countdown.


sub init()
    m.top.functionName = "RenderVideoContentWithAds"
    m.top.id = "PlayerTask"
    ' Set some field references.
    m.top.countdownSeconds = m.top.FindNode("countdownSeconds")
end sub

'''''''''
' RenderVideoContentWithAds: Plays the video content with ads.
' 
'''''''''
function RenderVideoContentWithAds() as void
    ' Define a bookmark object.
    m.playbackBookmark = Bookmark()
    ' Define an array for the playback items.
    m.playbackItems = []
    ' Get a reference to the video player.
    videoPlayer = m.top.video
    ' RAF needs a 'view' for displaying ads.
    view = videoPlayer.getParent()
    ' If the media content is an Episode, set the countdown seconds
    ' to trigger the building of the countdown framework.
    if m.top.mediaType = "episode"
        m.top.countdownSeconds = 0
    end if
    if videoPlayer.content.getChildCount() > 0
        m.playbackItems = videoPlayer.content.getChildren(- 1, 0)
    else
        m.playbackItems[0] = videoPlayer.content
    end if
    ' Get the reference to the Roku ad framework.
    RAF = Roku_Ads()
    ' Uncomment the next line for RAF debugging.
    'RAF.setDebugOutput(true)
    ' Set the Ad server URL.
    RAF.setAdUrl(m.global.AD_SERVER)
    RAF.enableAdMeasurements(true)
    ' Set the content genres and length.
    if m.playbackItems[0].genres <> invalid
        RAF.setContentGenre(m.playbackItems[0].genres)
    end if
    ' Set the length of the first item to play.
    RAF.setContentLength(int(m.playbackItems[0].length))
    ' Get the initial ad pods.
    adPods = RAF.getAds()
    ' Check for scheduled ad breaks.  Apply them if available.
    preRoll = [
        false
    ]
    adSchedule = []
    scheduledPods = ScheduleAdBreaks(RAF, m.playbackItems, adPods, preRoll, adSchedule)
    if scheduledPods.Count() > 0
        RAF.importAds(scheduledPods)
    end if
    ' Set a flag to indicate we should keep playing the video.
    keepPlaying = true
    ' Set the adbreak index.
    adBreakIndex = 0
    ' Show any pre-roll ads.
    if preRoll[0] = true
        totalAds = []
        adPods = GetAdPodsByType(scheduledPods, "preroll", totalAds)
        ctx = {
            start: 1
            total: totalAds[0]
        }
        keepPlaying = RAF.showAds(adPods, ctx, view)
        adBreakIndex++
    end if
    ' If the preroll rendered Ok, create the video message loop and set up the video object.
    port = CreateObject("roMessagePort")
    if keepPlaying = true
        videoPlayer.observeField("position", port)
        videoPlayer.observeField("contentIndex", port)
        videoPlayer.observeField("state", port)
        videoPlayer.control = "play"
        videoPlayer.setFocus(true)
    else
        ' End the task now.
        m.top.state = "stop"
    end if
    ' Initialize some variables.
    curPos = m.playbackItems[0].PlayStart
    videoPlayer.seek = curPos
    isPlayingPostroll = false
    stoppedForAdRendering = false
    ' Begin main video rendering loop.
    while keepPlaying
        msg = wait(0, port)
        if type(msg) = "roSGNodeEvent"
            ' VIDEO: POSITION.
            if msg.GetField() = "position"
                ' Get the current video position.
                curPos = msg.GetData()
                ' Get the time left for the video.
                timeleft = cint(videoPlayer.duration - videoPlayer.position)
                ' If the content is a playlist, check for the Episode countdown.
                if videoPlayer.contentIsPlaylist = true
                    if timeleft <= 10 AND timeleft >= 1
                        m.top.countdownSeconds = timeleft
                    end if
                end if
                if ShouldRenderAds(adBreakIndex, curPos, adSchedule)
                    'DebugPrint("PlayerTask:Position - Stopping video to play mid-roll adPod @ " + curPos.toStr())
                    ' Stop the video.  The video state will change and is handled @ 'Video state'.
                    videoPlayer.control = "stop"
                    videoPlayer.visible = false
                    stoppedForAdRendering = true
                end if
                ' VIDEO: CONTENTINDEX.
            else if msg.GetField() = "contentIndex"
                ' Get the index within the playlist of the currently playing video.
                contentIndex = msg.GetData()
                'DebugPrint("Video content index is now " + contentIndex.toStr())
                if contentIndex > 0
                    ' Force removal of Episode countdown from the screen.
                    m.top.countdownSeconds = - 1
                end if
                ' Check to see if the user is in the process of stopping video playback.
                if videoPlayer.state <> "stop" AND contentIndex <> 0
                    ' Get the prior video that just finished playing.
                    priorVideo = videoPlayer.content.getChild(contentIndex - 1)
                    ' Mark it as 'Watched'.
                    MarkVideoAsWatched(priorVideo.id)
                    ' Remove its Video bookmark.
                    ResetBookmark(priorVideo)
                    ' Set a bookmark for the current video.
                    currentVideo = videoPlayer.content.getChild(contentIndex)
                    videoBookmark = Bookmark()
                    videoBookmark.WriteNewEpisodeBookmark(currentVideo)
                    ' Set the global CurrentPlaybackContent.
                    m.global.CurrentPlaybackContent = currentVideo
                    videoPlayer.playbackId = currentVideo.id
                    ' If the content is a playlist, remove the first item.
                    ' NOTE: This is done because stopping and starting a playlist resets the
                    ' "contentIndex" field to zero.
                    if videoPlayer.contentIsPlaylist
                        videoPlayer.content.RemoveChildIndex(0)
                    end if
                end if
                ' VIDEO STATE.
            else if msg.GetField() = "state" then
                ' Get the current video state.
                curState = msg.GetData()
                ' Stopped.
                if curState = "stopped"
                    'DebugPrint("PlayerTask: Video playback stopped.")
                    ' If we're stopped for ad rendering, render the ad.
                    if stoppedForAdRendering
                        'DebugPrint("PlayerTask:State - Play scheduled mid-roll adPod " + adBreakIndex.toStr() + " at " + curPos.toStr())
                        ' Render the ad pod.
                        adPod = scheduledPods[adBreakIndex]
                        ctx = {
                            start: 1
                            total: adPod.ads.Count()
                        }
                        keepPlaying = RAF.showAds(adPod, ctx, view)
                        adBreakIndex++
                        ' If post-roll ad(s) are playing, exit while loop.
                        if isPlayingPostroll = true
                            exit while
                        end if
                        ' Continue playing video content if ad rendering was successful.
                        if keepPlaying = true
                            'DebugPrint("Mid-roll adPod finished, seeking to " + curPos.toStr())
                            videoPlayer.visible = true
                            videoPlayer.seek = curPos
                            videoPlayer.control = "play"
                            videoPlayer.setFocus(true)
                            stoppedForAdRendering = false
                        end if
                    else
                        ' We're stopped because the user pressed Back on the remote to end playback.
                        keepPlaying = false
                        m.top.state = "stop"
                    end if
                    ' Finished.
                else if curState = "finished" then
                    'DebugPrint("PlayerTask: Content has finished playing.")
                    ' Stop the video.  The video state will change and is handled @ 'Video state'.
                    videoPlayer.control = "stop"
                    videoPlayer.visible = false
                    if ShouldRenderAds(adBreakIndex, curPos, adSchedule)
                        'DebugPrint("PlayerTask: play scheduled post-roll ad.")
                        ' Set flag indicating post-roll ads are to be played.
                        isPlayingPostroll = true
                    else
                        keepPlaying = false
                        m.top.state = "done"
                    end if
                end if
            end if
        end if
    end while
    'DebugPrint("PlayerTask: exiting playContentWithAds()")
end function

'''''''''
' ScheduleAdBreaks: Schedule ad breaks if any.
' 
' @param {object} I: _RAF - reference to the RAF.
' @param {object} I: _playbackContent - the playback content.
' @param {object} I: _adPods - available ad pods.
' @param {object} O: _preRoll - flag indicating that preroll ads are present.
' @return {object} - returns an array of scheduled ads.
'''''''''
function ScheduleAdBreaks(_RAF as object, _playbackContent as object, _adPods as object, _preRoll as object, _adSchedule as object) as object
    ' Initialize an array to hold the scheduled ad pods.
    scheduledPods = []
    adBreaks = []
    ' Process each item in the playvback content.
    for each playbackItem in _playbackContent
        totalContentLength = playbackItem.length
        if playbackItem.mediaType <> "episode"
            adBreaks = playbackItem.adBreaks
        else
            adBreaks = playbackItem.content.adBreaks
        end if
        ' If adBreaks are requested, set up the schedule pod.
        if adBreaks <> invalid and adBreaks.Count() > 0
            ' Initialize the ad schedule.
            adBreakSchedule = []
            ' Get the scheduled ad breaks.
            for each adBreak in adBreaks
                adBreakSchedule.Push(adBreak)
            end for
            ' Scan each ad break and associate an ad pod with it.
            for each scheduledAdbreak in adBreakSchedule
                adBreakTime = ConvertAdbreakToSeconds(scheduledAdbreak)
                ' Check for preroll ads.
                if adBreakTime = 0
                    ' Set a flag indicating there are preroll ads.
                    _preroll[0] = true
                    ' Now find the preroll ads to render.
                    for each adPod in _adPods
                        if adPod.renderSequence = "preroll"
                            scheduledPods.Push({
                                viewed: false
                                renderSequence: adPod.renderSequence
                                duration: adPod.duration
                                renderTime: adBreakTime
                                ads: adPod.ads
                                contentId: playbackItem.id
                            })
                            _adSchedule.Push({
                                type: "preroll"
                                adbreakTime: adBreakTime
                                contentId: playbackItem.id
                            })
                        end if
                    end for
                    ' Check for midroll ads.
                else if adBreakTime <> totalContentLength
                    ' Now find the midroll ads to render.
                    for each adPod in _adPods
                        if adPod.renderSequence = "midroll"
                            scheduledPods.Push({
                                viewed: false
                                renderSequence: adPod.renderSequence
                                duration: adPod.duration
                                renderTime: adBreakTime
                                ads: adPod.ads
                                contentId: playbackItem.id
                            })
                            _adSchedule.Push({
                                type: "midroll"
                                adBreakTime: adBreakTime
                                contentId: playbackItem.id
                            })
                        end if
                    end for
                else if adBreakTime >= totalContentLength
                    ' Now find the postroll ads to render.
                    for each adPod in _adPods
                        if adPod.renderSequence = "postroll"
                            scheduledPods.Push({
                                viewed: false
                                renderSequence: adPod.renderSequence
                                duration: adPod.duration
                                renderTime: adBreakTime
                                ads: adPod.ads
                                contentId: playbackItem.id
                            })
                            _adSchedule.Push({
                                type: "postroll"
                                adBreakTime: ConvertAdbreakToSeconds(scheduledAdbreak)
                                contentId: playbackItem.id
                            })
                        end if
                    end for
                end if
            end for
        end if
    end for
    return scheduledPods
end function

'''''''''
' ShouldRenderAds: Determined if ads should be played at the specified position.
' 
' @param {integer} I: _adbreakIndex - the index of the current ad.
' @param {integer} I: _curPos - the current time position.
' @param {object} I: _adSchedule - the ad schedule.
' @return {boolean} - returns true if ads should be rendered, else, false.
'''''''''
function ShouldRenderAds(_adbreakIndex as integer, _curPos as integer, _adSchedule as object) as boolean
    renderAd = false
    if _adbreakIndex < _adSchedule.Count()
        potentialAdBreak = _adSchedule[_adbreakIndex]
        if potentialAdBreak.contentId = m.global.CurrentPlaybackContent.id AND _curPos >= potentialAdBreak.adBreakTime
            renderAd = true
        end if
    end if
    return renderAd
end function

'''''''''
' GetAdPodsByType: Gets ad pods by type.
' 
' @param {object} I: _adPodsArray - an array of adpods.
' @param {string} I: _adType - the type of ad pod to find.
' @param {object} O: _totalAds - the total number of ads in the returned ad pods.
' @return {object} - returns an ad pod array.
'''''''''
function GetAdPodsByType(_adPodsArray as object, _adType as string, _totalAds as object) as object
    adPodArray = []
    totalAds = 0
    for each adPod in _adPodsArray
        if adPod.renderSequence = _adType AND m.global.CurrentPlaybackContent.id = adPod.contentId
            totalAds += adPod.ads.Count()
            adPodArray.Push(adPod)
        end if
    end for
    _totalAds[0] = totalAds
    return adPodArray
end function

'''''''''
' ResetBookmark: Resets the content bookmark by removing it.
' 
' @param {object} I: _feedContent - the feed content.
' @return {dynamic}
'''''''''
function ResetBookmark(_feedContent as object)
    ' Get the Bookmark object.
    m.contentBookmark = Bookmark()
    ' Reset the playback position.
    _feedContent.PlayStart = 0
    ' Remove the bookmark.
    m.contentBookmark.DeleteBookmark(_feedContent.id)
end function'//# sourceMappingURL=./PlayerTaskWithAds.bs.map