' Define the Continue Watching row title.



sub Init()
    ' set the name of the function in the Task node component to be executed when the state field changes to RUN.
    ' in our case, this method executes after the following cmd: m.contentTask.control = "run"(see Init method in MainScene)
    m.top.functionName = "GetContent"
    ' Create a DateTime object for later use.
    m.dateObject = CreateObject("roDateTime")
end sub

'''''''''
' BuildContinueWatching: Builds the Continue Watching row in the Grid.
' 
' @param {object} I: _root - the root object of the Grid.
' @param {string} I: [_feedType="Json"] - the feed type.
' @return {dynamic}
'''''''''
function BuildContinueWatching(_root as object, _feedType = "Json" as string)
    bookmarks = Bookmark()
    cWatchingContentIds = bookmarks.GetKeyList()
    ' Nothing to do if no resumable content.
    if cWatchingContentIds.Count() = 0
        return invalid
    end if
    row = {}
    row.title = "Continue watching"
    row.children = []
    itemData = invalid
    contentType = []
    for each item in cWatchingContentIds
        if _feedType = "Json"
            contentObject = QueryJsonObjectById(item, contentType)
            if contentObject <> invalid
                itemData = GetJsonItemData(contentObject, contentType[0])
                if itemData.mediaType = "episode"
                    seasonIndex = []
                    episodeIndex = GetEpisodeIndex(itemData, seasonIndex)
                    itemData.episodeIndex = episodeIndex
                    if seasonIndex.Count() > 0
                        itemData.parentSeasonindex = seasonIndex[0]
                    end if
                    itemData.playAll = true
                end if
            end if
        else
            contentObject = QueryXmlObjectById(item)
            if contentObject <> invalid
                itemData = GetXmlItemData(contentObject)
            end if
        end if
        if itemData <> invalid
            itemData["continueWatching"] = true
            row.children.Push(itemData)
        end if
    end for
    ' Insert as first item in the Grid.
    if row.children.Count() > 0
        _root.Unshift(row)
    end if
end function

'''''''''
' ParseJsonFeed: Retrieves the JSON feed file from the URL and parses it.
' 
' @param {string} I: feedURL - the URL of the feed.
'''''''''
sub ParseJsonFeed(feedURL as string)
    ' request the content feed from the API.
    xfer = CreateObject("roURLTransfer")
    xfer.SetCertificatesFile("common:/certs/ca-bundle.crt")
    xfer.SetURL(feedURL)
    rsp = xfer.GetToString()
    ' Parent of channel rows.
    rootChildren = []
    ' Content Search index.
    m.contentIndex = []
    ' parse the JSON content feed and build a tree of ContentNodes to populate the GridView.
    m.jsonContentFeed = ParseJson(rsp)
    ' Save a copy of the feed file in 'tmp:/'.
    WriteAsciiFile("tmp:/channelFeed.json", FormatJson(m.jsonContentFeed))
    if m.jsonContentFeed <> invalid
        ' Define an array of rows that contain the Categories from the feed.
        if m.jsonContentFeed.categories <> invalid
            for each category in m.jsonContentFeed.categories
                row = {}
                row.title = category.Lookup("name")
                row.order = category.Lookup("order")
                row.children = []
                rootChildren.Push(row)
            end for
        end if
        ' Parse audio-only objects from the feed file.
        if m.jsonContentFeed.audioOnly <> invalid
            for each audioStream in m.jsonContentFeed.audioOnly
                ' Add this item to the Search index.
                IndexThisItem(audioStream, "audioOnly")
                if ObjectContainsTags(audioStream) = true
                    categories = {}
                    categories = QueryCategoryForTags(audioStream.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(audioStream, "audioOnly")
                                        row.children.Push(itemData)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + audioStream.title)
                                        sleep(3000)
                                    end try
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse movie objects from the feed file.
        if m.jsonContentFeed.movies <> invalid
            for each movie in m.jsonContentFeed.movies
                ' Add this item to the Search index.
                IndexThisItem(movie, "movies")
                if ObjectContainsTags(movie) = true
                    categories = {}
                    categories = QueryCategoryForTags(movie.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(movie, "movie")
                                        row.children.Push(itemData)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + movie.title)
                                        sleep(3000)
                                    end try
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse liveFeed objects from the feed file.
        if m.jsonContentFeed.liveFeeds <> invalid
            for each liveFeed in m.jsonContentFeed.liveFeeds
                ' Add this item to the Search index.
                IndexThisItem(liveFeed, "livefeeds")
                if ObjectContainsTags(liveFeed) = true
                    categories = {}
                    categories = QueryCategoryForTags(liveFeed.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(liveFeed, "livefeed")
                                        row.children.Push(itemData)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + liveFeed.title)
                                        sleep(3000)
                                    end try
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse series objects from the feed file.
        if m.jsonContentFeed.series <> invalid
            for each series in m.jsonContentFeed.series
                ' Add this item to the Search index.
                IndexThisItem(series, "series")
                if ObjectContainsTags(series) = true
                    categories = {}
                    categories = QueryCategoryForTags(series.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(series, "series", true)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + series.title)
                                        sleep(3000)
                                    end try
                                    AppendSeriesItemData(itemData, series)
                                    row.children.Push(itemData)
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse shortFormVideos objects from the feed file.
        if m.jsonContentFeed.shortFormVideos <> invalid
            for each shortFormVideo in m.jsonContentFeed.shortFormVideos
                ' Add this item to the Search index.
                IndexThisItem(shortFormVideo, "shortFormVideos")
                if ObjectContainsTags(shortFormVideo) = true
                    categories = {}
                    categories = QueryCategoryForTags(shortFormVideo.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(shortFormVideo, "shortFormVideo")
                                        row.children.Push(itemData)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + shortFormVideo.title)
                                        sleep(3000)
                                    end try
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse tvSpecials objects from the feed file.
        if m.jsonContentFeed.tvSpecials <> invalid
            for each tvSpecial in m.jsonContentFeed.tvSpecials
                ' Add this item to the Search index.
                IndexThisItem(tvSpecial, "tvSpecials")
                if ObjectContainsTags(tvSpecial) = true
                    categories = {}
                    categories = QueryCategoryForTags(tvSpecial.tags)
                    if categories <> invalid
                        for each categoryName in categories
                            for each row in rootChildren
                                if row.title = categoryName
                                    try
                                        ' Build assoc array for the row.
                                        itemData = GetJsonItemData(tvSpecial, "tvSpecial")
                                        row.children.Push(itemData)
                                    catch excp
                                        UpdateLoadingIndicator("ERROR! Could not load item " + tvSpecial.title)
                                        sleep(3000)
                                    end try
                                end if
                            end for
                        end for
                    end if
                end if
            end for
        end if
        ' Parse playlists objects from the feed file.
        if m.jsonContentFeed.playlists <> invalid
            playlistObjectType = []
            for each playlist in m.jsonContentFeed.playlists
                category = GetCategoryFromPlaylist(playlist.name)
                if category <> invalid
                    if playlist.itemIds <> invalid
                        for each id in playlist.itemIds
                            playlistObjectType[0] = invalid
                            playlistObject = QueryJsonObjectById(id, playlistObjectType)
                            if playlistObject <> invalid
                                row = QueryChannelRowByCategoryName(category.name, rootChildren)
                                try
                                    ' Build assoc array for the row. 
                                    if playlistObjectType[0] = "series"
                                        itemData = GetJsonItemData(playlistObject, "series", true)
                                        AppendSeriesItemData(itemData, playlistObject)
                                    else
                                        itemData = GetJsonItemData(playlistObject, playlistObjectType[0])
                                    end if
                                    row.children.Push(itemData)
                                catch excp
                                    UpdateLoadingIndicator("ERROR! Could not load item " + playlistObject.title)
                                    sleep(3000)
                                end try
                            end if
                        end for
                    end if
                end if
            end for
            row.title = "Playlist: " + row.title
        end if
        showContinueWatchingRow = m.global.CONTINUE_WATCHING
        if showContinueWatchingRow
            BuildContinueWatching(rootChildren)
        end if
        ' Check for a parsing error.
        if Type(row) = "<uninitialized>"
            UpdateLoadingIndicator("ERROR! Feed items could not be parsed.")
            Sleep(3000)
        end if
        ' Sort each row on 'releaseDate'
        for each rowChild in rootChildren
            ' Get the child items in the grid row.
            rowChildren = rowChild.children
            ' If the row contains children and the order is NOT 'manual'.
            if rowChildren.Count() > 0 AND rowChild.order <> "manual"
                if rowChild.order = "most_recent"
                    rowChildren.SortBy("releaseDate", "r")
                else if rowChild.order = "chronological"
                    rowChildren.SortBy("releaseDate")
                end if
            end if
        end for
        ' Save the Search indices to the global node.
        m.global.addFields({
            "ContentIndex": m.contentIndex
        })
        ' set up a root ContentNode to represent rowList on the GridScreen.
        contentNode = CreateObject("roSGNode", "ContentNode")
        contentNode.Update({
            children: rootChildren
        }, true)
        ' populate content field with root content node.
        ' Observer(see OnMainContentLoaded in ContentTaskLogic.brs) is invoked at this moment.
        m.top.content = contentNode
    else
        UpdateLoadingIndicator("ERROR! Parsing JSON feed file.")
    end if
end sub

'''''''''
' IndexThisItem: Builds a Search index for the channel.
' 
' @param {object} I: _contentObject - the content object to index.
' @param {string} I: _mediaType - the content type.
'''''''''
function IndexThisItem(_contentObject as object, _mediaType as string) as void
    if _mediaType <> "series"
        description = (function(_contentObject)
                __bsConsequent = _contentObject.longDescription
                if __bsConsequent <> invalid then
                    return __bsConsequent
                else
                    return _contentObject.shortDescription
                end if
            end function)(_contentObject)
        indexEntry = {
            id: _contentObject.id
            title: _contentObject.title
            longDescription: description
            mediaType: _mediaType
            tags: (function(__bsCondition, _contentObject)
                    if __bsCondition then
                        return _contentObject.tags
                    else
                        return invalid
                    end if
                end function)(_contentObject.tags <> invalid, _contentObject)
        }
        m.contentIndex.Push(indexEntry)
    else
        ' Save the Series object.
        description = (function(_contentObject)
                __bsConsequent = _contentObject.longDescription
                if __bsConsequent <> invalid then
                    return __bsConsequent
                else
                    return _contentObject.shortDescription
                end if
            end function)(_contentObject)
        indexEntry = {
            id: _contentObject.id
            title: _contentObject.title
            longDescription: description
            mediaType: _mediaType
            tags: (function(__bsCondition, _contentObject)
                    if __bsCondition then
                        return _contentObject.tags
                    else
                        return invalid
                    end if
                end function)(_contentObject.tags <> invalid, _contentObject)
        }
        m.contentIndex.Push(indexEntry)
        ' Save the Episodes within a Season.
        if _contentObject.seasons <> invalid
            for each season in _contentObject.seasons
                ' Initialize an Episode index.
                episodeIndex = 0
                for each episode in season.episodes
                    description = (function(episode)
                            __bsConsequent = episode.longDescription
                            if __bsConsequent <> invalid then
                                return __bsConsequent
                            else
                                return episode.shortDescription
                            end if
                        end function)(episode)
                    indexEntry = {
                        id: episode.id
                        series: _contentObject.id
                        season: season.title
                        title: episode.title
                        episodeIndex: episodeIndex
                        longDescription: description
                        mediaType: "episode"
                    }
                    m.contentIndex.Push(indexEntry)
                    episodeIndex++
                end for
            end for
        else
            ' Save the Episodes within a Series.
            ' Initialilze an Episode index.
            episodeIndex = 0
            for each episode in _contentObject.episodes
                description = (function(episode)
                        __bsConsequent = episode.longDescription
                        if __bsConsequent <> invalid then
                            return __bsConsequent
                        else
                            return episode.shortDescription
                        end if
                    end function)(episode)
                indexEntry = {
                    id: episode.id
                    series: _contentObject.id
                    title: episode.title
                    episodeIndex: episodeIndex
                    longDescription: description
                    mediaType: "episode"
                }
                m.contentIndex.Push(indexEntry)
                episodeIndex++
            end for
        endif
    end if
end function

'''''''''
' AppendSeriesItemData: Appends more data to the Series item.
' 
' @param {object} I/O:_itemData - the itemData to be updated.
' @param {object} I: _series - the Series that provides data to update.
' @return {dynamic}
'''''''''
function AppendSeriesItemData(_itemData as object, _series as object)
    if _series.seasons <> invalid
        seasons = GetSeasonData(_series)
        _itemData.numSeasons = _series.seasons.Count()
        numEpisodes = 0
        for each season in seasons
            numEpisodes += season.childEpisodes.Count()
        end for
        _itemData.numEpisodes = numEpisodes
    end if
    if _series.seasons <> invalid and _series.seasons.Count() > 0
        _itemData.children = seasons
    end if
    if _series.episodes <> invalid and _series.episodes.Count() > 0
        _itemData.childEpisodes = GetEpisodeData(_series)
        _itemData.numEpisodes = _itemData.childEpisodes.Count()
    end if
end function

'''''''''
' ParseXMLFeed: Retrieves the XML feed file from the URL and parses it.
' 
' @param {string} I: feedURL - the URL of the feed.
'''''''''
sub ParseXMLFeed(feedURL as string)
    ' request the content feed from the API.
    xfer = CreateObject("roURLTransfer")
    xfer.SetCertificatesFile("common:/certs/ca-bundle.crt")
    xfer.SetURL(feedURL)
    rsp = xfer.GetToString()
    ' Parent of channel rows.
    rootChildren = []
    ' Array of channel Categories.
    categories = []
    m.xmlObject = CreateObject("roXMLElement")
    if m.xmlObject.Parse(rsp) = true
        ' Define an AA array to keep track of the found categories. 
        uniqueCategories = {}
        ' Get the Channel object.
        channel = m.xmlObject.GetNamedElements("channel")
        ' Get all of the Channel Items.
        items = channel.GetNamedElements("item")
        ' Get the count of Items.
        count = items.Count()
        ' Set the list index to the top.
        items.ResetIndex()
        ' Iterate the list of Channel Items to extract Categories.
        for index = 0 to count - 1
            ' Get an Item tag.
            itemTag = items.GetIndex()
            ' Get the associated Category tags.
            categoryTags = itemTag.GetNamedElements("media:category")
            ' Get the count of Categories.
            categoryCount = categoryTags.Count()
            ' Set the list index to the top.
            categoryTags.ResetIndex()
            for categoryIndex = 0 to categoryCount - 1
                ' Get the category item.
                categoryItem = categoryTags.GetIndex()
                ' Get the Category text and save it.
                category = categoryItem.GetText()
                ' Make sure we haven't already added the category to the row.
                if uniqueCategories.doesExist(category)
                else
                    categories.Push(category)
                    uniqueCategories[category] = true
                    row = {}
                    row.title = category
                    row.children = []
                    rootChildren.Push(row)
                end if
            end for
        end for
        ' Set the list index to the top.
        items.ResetIndex()
        ' Iterate the list of Channel Items to bucket them.
        for index = 0 to count - 1
            ' Get an Item tag.
            itemTag = items.GetIndex()
            ' Get the associated Category tags.
            categoryTags = itemTag.GetNamedElements("media:category")
            ' Get the count of Categories.
            categoryCount = categoryTags.Count()
            ' Set the list index to the top.
            categoryTags.ResetIndex()
            for categoryIndex = 0 to categoryCount - 1
                ' Get the category item.
                categoryItem = categoryTags.GetIndex()
                ' Get the Category text and save it.
                category = categoryItem.GetText()
                for each row in rootChildren
                    if row.title = category
                        ' Build assoc array for the row.
                        itemData = GetXmlItemData(itemTag)
                        row.children.Push(itemData)
                    end if
                end for
            end for
        end for
        cwRow = m.global.CONTINUE_WATCHING
        if cwRow <> invalid and cwRow = "true"
            BuildContinueWatching(rootChildren, "xml")
        end if
        ' Add all Items to the row.
        rootChildren.Push(row)
        ' Check for a parsing error.
        if Type(row) = "<uninitialized>"
            UpdateLoadingIndicator("ERROR! Feed items could not be parsed.")
            Sleep(3000)
        end if
        ' Sort each row on 'releaseDate'
        for each rowChild in rootChildren
            ' Get the child items in the grid row.
            rowChildren = rowChild.children
            ' If the row contains children sort it.
            if rowChildren.Count() > 0
                rowChildren.SortBy("releaseDate")
            end if
        end for
        ' set up a root ContentNode to represent rowList on the GridScreen.
        contentNode = CreateObject("roSGNode", "ContentNode")
        contentNode.Update({
            children: rootChildren
        }, true)
        ' populate content field with root content node.
        ' Observer(see OnMainContentLoaded in ContentTaskLogic.brs) is involed at this moment.
        m.top.content = contentNode
    else
        UpdateLoadingIndicator("ERROR! Parsing XML feed file.")
    end if
end sub

'''''''''
' GetContent: Gets the content feed and populates the rowList.
' 
'''''''''
sub GetContent()
    ' Get the feed URL.
    feedURL = m.global.FEED_URL
    ' Convert to lowercase.
    lfeedURL = LCase(feedURL)
    jsonIndex = Instr(1, lfeedURL, ".json")
    if jsonIndex <> 0
        ParseJsonFeed(feedURL)
    else
        xmlIndex = Instr(1, lfeedURL, ".xml")
        if xmlIndex <> 0
            ParseXMLFeed(feedURL)
        else
            xmlIndex = Instr(1, lfeedURL, ".rss")
            if xmlIndex <> 0
                ParseXMLFeed(feedURL)
            else
                UpdateLoadingIndicator("ERROR! Missing file extension. Cannot determine feed file type.")
            end if
        end if
    end if
end sub

'''''''''
' GetXmlItemData: Gets the XML item data.
' 
' @param {object} I: channelItem - an Item object.
' @return {object} - returns a playable item.
'''''''''
function GetXmlItemData(channelItem as object) as object
    item = {}
    ' Set the object type.
    item.mediaType = "shortFormVideo"
    ' Get the child nodes of the Item.
    nodes = channelItem.GetChildNodes()
    ' Iterate through the child nodes of the Item.
    for each node in nodes
        ' Get the node name.
        name = node.GetName()
        ' Get its value.
        value = node.GetText()
        if name = "guid"
            item.id = value
        end if
        if name = "media:description"
            item.longDescription = value
            item.shortDescription = value
            item.description = value
        end if
        if name = "media:thumbnail"
            attrs = node.GetAttributes()
            item.hdPosterURL = attrs["url"]
        end if
        if name = "media:title"
            item.title = value
        end if
        if name = "pubDate"
            m.dateObject.FromISO8601String(value)
            releaseDate = m.dateObject.asDateStringLoc("y-MM-dd")
            item.releaseDate = releaseDate
        end if
        if name = "media:content"
            attrs = node.GetAttributes()
            item.length = attrs["duration"]
            item.Url = attrs["url"]
        end if
        ' CAPTION FILES.
        if name = "media:subTitle"
            attrs = node.GetAttributes()
            tracks = {}
            item.SubtitleUrl = attrs["href"]
            tracks["Language"] = attrs["lang"]
            tracks["TrackName"] = attrs["href"]
            item.SubtitleTracks = tracks
        end if
    end for
    ' Default the video start position to 0.0
    item.PlayStart = 0.0
    return item
end function

'''''''''
' GetJsonItemData: Gets the item content as an AA.
' 
' @param {object} I: contentObject - the feed content object.
' @param {string} I: contentType - the feed content type.
' @param {boolean} I: [isSeries=false] - flag to indicate a Series.
' @return {object} - returns a playable item.
'''''''''
function GetJsonItemData(contentObject as Object, contentType as string, isSeries = false as boolean) as Object
    ' Initialize a content item.
    item = {}
    ' Initialize the item's genres.
    genres = []
    ' Initialize the item's adBreaks.
    adbreaks = []
    ' Set the object type.
    item.mediaType = contentType
    ' populate some standard content metadata fields to be displayed on the GridScreen
    ' CONTENT ID.
    item.id = contentObject.id
    ' LONG DESCRIPTION and SHORT DESCRIPTION.
    if contentObject.longDescription <> invalid
        item.longDescription = contentObject.longDescription
    end if
    if contentObject.shortDescription <> invalid
        item.shortDescription = contentObject.shortDescription
    end if
    if item.longDescription <> invalid
        item.description = item.longDescription
    else if item.shortDescription <> invalid
        item.description = item.shortDescription
    end if
    ' THUMBNAIL.
    item.hdPosterURL = contentObject.thumbnail
    ' SEASON NUMBER.
    ' If the content does NOT contain a 'seasonNumber' field, it's not a Series/Season object.
    if contentObject.seasonNumber = invalid
        item.title = contentObject.title
    else
        item.title = "Season " + contentObject.seasonNumber.ToStr()
    end if
    ' RELEASE DATE.
    item.releaseDate = contentObject.releaseDate
    'EPISODE NUMBER.
    if contentObject.episodeNumber <> invalid
        item.episodePosition = contentObject.episodeNumber.ToStr()
        item.episodeNumber = contentObject.episodeNumber
    end if
    ' DURATION.
    if contentObject.content <> invalid
        ' populate length of content to tbe displayed on the GridScreen.
        item.length = contentObject.content.duration
        item.Url = contentObject.content.videos[0].Url
    end if
    ' GENRES.
    if contentObject.genres <> invalid and contentObject.genres.Count() > 0
        for each genre in contentObject.genres
            genres.Push(genre)
        end for
        item.genres = genres
    end if
    ' ADBREAKS.
    if isSeries = false
        if contentObject.content <> invalid
            if contentObject.content.adBreaks <> invalid and contentObject.content.adBreaks.Count() > 0
                for each adbreak in contentObject.content.adBreaks
                    adbreaks.Push(adbreak)
                end for
                item.adBreaks = adbreaks
            end if
        end if
    end if
    ' TRICKPLAY FILES.
    if isSeries = false and contentType <> "season"
        if contentObject.content.trickPlayFiles <> invalid and contentObject.content.trickPlayFiles.Count() > 0
            for each tpf in contentObject.content.trickPlayFiles
                if tpf.quality = "HD"
                    item.HDBifUrl = tpf.url
                else if tpf.quality = "SD"
                    item.SDBifUrl = tpf.url
                end if
            end for
        end if
    end if
    ' CAPTION FILES.
    if isSeries = false and contentType <> "season"
        if contentObject.content.captions <> invalid and contentObject.content.captions.Count() > 0
            subtitleTracks = []
            for each caption in contentObject.content.captions
                track = {}
                track["Language"] = caption.language
                track["TrackName"] = caption.url
                subtitleTracks.Push(track)
            end for
            item.SubtitleTracks = subtitleTracks
        end if
    end if
    ' CREDITS.
    if contentType <> "season"
        if contentObject.credits <> invalid and contentObject.credits.Count() > 0
            Actors = []
            Hosts = []
            Narrators = []
            Directors = []
            Producers = []
            for each credit in contentObject.credits
                if credit.role = "actor"
                    Actors.Push(credit.name)
                else if credit.role = "host"
                    Hosts.Push(credit.name)
                else if credit.role = "narrator"
                    Narrators.Push(credit.name)
                else if credit.role = "director"
                    Directors.Push(credit.name)
                else if credit.role = "producer"
                    Producers.Push(credit.name)
                end if
            end for
            item.Actors = Actors
            item.Hosts = Hosts
            item.Narrators = Narrators
            item.Directors = Directors
            item.Producers = Producers
        end if
    end if
    ' RATING.
    if contentObject.rating <> invalid
        item.ratingObj = contentObject.rating
    end if
    ' SERIES/SEASONS.    
    if isSeries = true and contentObject.seasons <> invalid
        item.hasSeasons = true
    else
        item.hasSeasons = false
    end if
    ' Default the video start position to 0.0
    item.PlayStart = 0.0
    return item
end function

'''''''''
' GetEpisodeData: Gets the Episodes data.
' 
' @param {object} I: _series - the Series conent.
' @return {object} = returns an array of Series Episodes.
'''''''''
function GetEpisodeData(_series as Object) as object
    ' Init some arrays.
    episodeArray = []
    for each episode in _series.episodes
        episodeData = GetJsonItemData(episode, "episode")
        genresArray = []
        ' Embed the Series genres in the Episode.
        for each genre in _series.genres
            genresArray.Push(genre)
        end for
        ' save episode data items.
        episodeData.titleEpisode = episode.title
        episodeData.episodeNumber = episode.episodeNumber
        episodeData.episodeIndex = 0
        episodeData.thumbnail = episode.thumbnail
        episodeData.genres = genresArray
        episodeData.mediaType = "episode"
        episodeData.parentSeries = _series.id
        episodeArray.Push(episodeData)
    end for
    ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
    ' For 'chronological order, remove the 'r' parameter.
    episodeArray.SortBy("releaseDate", "r")
    ' Reindex the Episodes since they've been sorted.
    ReindexEpisodes(episodeArray)
    return episodeArray
end function

'''''''''
' GetSeasonData: Gets the Season data.
' 
' @param {object} I: _series - the Series object.
' @return {object} - returns the Season data.
'''''''''
function GetSeasonData(_series as Object) as Object
    ' Get the Season data.
    seasons = _series.seasons
    seasonsArray = []
    if seasons <> invalid
        ' Iterate the Seasons.
        for each season in seasons
            if season.episodes <> invalid
                episodes = []
                for each episode in season.episodes
                    episodeData = GetJsonItemData(episode, "episode")
                    ' save season title for element to represent it on the episodes screen
                    episodeData.titleSeason = season.title
                    episodeData.episodeNumber = episode.episodeNumber
                    ' Set a temporary index number.
                    episodeData.episodeIndex = 0
                    episodeData.mediaType = "episode"
                    episodeData.parentSeries = _series.id
                    episodeData.parentSeasonIndex = season.seasonNumber.toStr()
                    episodes.Push(episodeData)
                end for
                ' Sort Episodes on 'releaseDate' field (effectively, 'most_recent' order).
                episodes.SortBy("releaseDate", "r")
                ' Reindex the Episodes since they have been sorted.
                ReindexEpisodes(episodes)
                seasonData = GetJsonItemData(season, "season")
                ' populate season's childEpisodes field with its episodes
                ' as a result season's ContentNode will contain episode's nodes
                seasonData.childEpisodes = episodes
                ' set content type for season object to represent it on the screen as section with episodes
                seasonData.contentType = "section"
                seasonsArray.Push(seasonData)
            end if
        end for
    end if
    return seasonsArray
end function

'''''''''
' ObjectContainsTags: Checks the object for category tags.
' 
' @param {object} obj
' @return {boolean} true if object contains tags, else, fale.
'''''''''
function ObjectContainsTags(obj as Object) as boolean
    if obj.tags <> invalid
        return true
    else
        return false
    end if
end function

'''''''''
' arrayContainsValue: Checks an array for a value match.
' 
' @param {object} I: _array - the array to check.
' @param {string} I: value - the value to check.
' @return {boolean} - returns true if the value is found in the array, else false.
'''''''''
function arrayContainsValue(_array as object, value as String) as boolean
    for each item in _array
        item = LCase(item)
        item = TrimAString(item)
        value = LCase(value)
        value = TrimAString(value)
        if item = value
            return true
        end if
    end for
    return false
end function

'''''''''
' QueryCategoryForTags: Query categories for the passed tags.
' 
' @param {object} I: tags - array of convent tags.
' @return {object} - returns an array of categories that specify the tags.
'''''''''
function QueryCategoryForTags(tags as Object) as Object
    categories = []
    ' Scan all categories.
    for each category in m.jsonContentFeed.categories
        ' If the query field is valid...
        if category.query <> invalid
            ' Does the query contain 'AND'?
            andRegxp = CreateObject("roRegex", "\band\b", "i")
            andQuery = andRegxp.IsMatch(LCase(category.query))
            ' Does the query contain 'OR'?
            orRegxp = CreateObject("roRegex", "\bor\b", "i")
            orQuery = orRegxp.IsMatch(LCase(category.query))
            if andQuery = true
                ' Get all the tags in the query.
                tagsInQuery = andRegxp.Split(LCase(category.query))
                ' Assume all of the input tags are present.
                allTagsFound = true
                ' Scan the query tags for a match in the input tags.
                for each tag in tagsInQuery
                    if arrayContainsValue(tags, LCase(tag)) = false
                        allTagsFound = false
                        exit for
                    end if
                end for
                ' If all tags were found in the query, save the category name.
                if allTagsFound = true
                    categories.Push(category.name)
                end if
            else if orQuery = true
                ' Get all the tags in the query.
                tagsInQuery = orRegxp.Split(LCase(category.query))
                ' Assume at least one of the input tags are present.
                aTagIsFound = false
                ' Scan the input tags for a match in the query tags.
                for each tag in tags
                    if arrayContainsValue(tagsInQuery, LCase(tag)) = true
                        aTagIsFound = true
                        exit for
                    end if
                end for
                ' If any tag was found in the query, save the category name.
                if aTagIsFound = true
                    categories.Push(category.name)
                end if
            else
                ' Scan all tags for a match in the category.
                for each tag in tags
                    if LCase(tag) = LCase(category.query)
                        categories.Push(category.name)
                    end if
                end for
            end if
        else
            continue for
        end if
    end for
    return categories
end function

'''''''''
' GetCategoryFromPlaylist: Get the catgegory object from the playlist name.
' 
' @param {string} playlistname
' @return {object}  a category object.
'''''''''
function GetCategoryFromPlaylist(playlistname as string) as Object
    foundcategory = invalid
    if m.jsonContentFeed.categories <> invalid
        for each category in m.jsonContentFeed.categories
            if playlistname = category.playlistName
                foundcategory = category
                exit for
            end if
        end for
    end if
    return foundcategory
end function

'''''''''
' QueryXmlObjectById: Query an XML feed object by its id. 
' 
' @param {string} I: _id - id of the feed content.
' @return {object} - returns the found feed object.
'''''''''
function QueryXmlObjectById(_id as string) as object
    object = invalid
    rexp = Substitute("\b{0}\b", _id)
    regxp = CreateObject("roRegex", rexp, "i")
    channel = m.xmlObject.GetNamedElements("channel")
    ' Get all of the Channel Items.
    items = channel.GetNamedElements("item")
    for each item in items
        guid = item.GetNamedElements("guid")
        id = guid[0].GetText()
        if regxp.IsMatch(id)
            object = item
            exit for
        end if
    end for
    return object
end function

'''''''''
' QueryJsonObjectById: Query a Json feed object by its id.
' 
' @param {string} I: id - id of the feed object.
' @param {object} O: contentType - the found content type.
' @return {object} - returns the found feed object.
'''''''''
function QueryJsonObjectById(id as string, contentType as object) as Object
    object = invalid
    rexp = Substitute("\b{0}\b", id)
    regxp = CreateObject("roRegex", rexp, "i")
    ' Parse movies.
    if m.jsonContentFeed.movies <> invalid
        for each movie in m.jsonContentFeed.movies
            ' If the passed id matches the movie id, we found the object.
            if regxp.IsMatch(movie.id)
                contentType[0] = "movies"
                return movie
            end if
        end for
    end if
    ' Parse liveFeeds.
    if m.jsonContentFeed.liveFeeds <> invalid
        for each liveFeed in m.jsonContentFeed.liveFeeds
            ' If the passed id matches the liveFeed id, we found the object.
            if regxp.IsMatch(liveFeed.id)
                contentType[0] = "liveFeeds"
                return liveFeed
            end if
        end for
    end if
    ' Parse Series, Seasons and Episodes.
    if m.jsonContentFeed.series <> invalid
        for each serie in m.jsonContentFeed.series
            ' If the passed id matches the Series id, we found the object.
            if regxp.IsMatch(serie.id)
                contentType[0] = "series"
                return serie
            end if
        end for
        for each serie in m.jsonContentFeed.series
            if serie.seasons <> invalid and serie.seasons.Count() > 0
                for each season in serie.seasons
                    episodes = season.episodes
                    for each episode in episodes
                        ' If the passed id matches the Episode id, we found the object.
                        if regxp.IsMatch(episode.id)
                            contentType[0] = "episode"
                            episode.parentSeries = serie.id
                            return episode
                        end if
                    end for
                end for
            else
                episodes = serie.episodes
                for each episode in episodes
                    ' If the passed id matches the Episode id, we found the object.
                    if regxp.IsMatch(episode.id)
                        contentType[0] = "episode"
                        episode.parentSeries = serie.id
                        return episode
                    end if
                end for
            end if
        end for
    end if
    ' Parse shortFormVideos.
    if m.jsonContentFeed.shortFormVideos <> invalid
        for each shortFormVideo in m.jsonContentFeed.shortFormVideos
            ' If the passed id matches the shortFormVideo id, we found the object.
            if regxp.IsMatch(shortFormVideo.id)
                contentType[0] = "shortFormVideos"
                return shortFormVideo
            end if
        end for
    end if
    ' Parse tvSpecials.
    if m.jsonContentFeed.tvSpecials <> invalid
        for each tvSpecial in m.jsonContentFeed.tvSpecials
            ' If the passed id matches the tvSpecial id, we found the object.
            if regxp.IsMatch(tvSpecial.id)
                contentType[0] = "tvSpecials"
                return tvSpecial
            end if
        end for
    end if
    return object
end function

'''''''''
' QueryChannelRowByCategoryName: Query the channel row for a matching category.
' 
' @param {string} name
' @param {object} rootChildren
' @return {object}  The corresponding row of the category.
'''''''''
function QueryChannelRowByCategoryName(name as string, rootChildren as object) as object
    for each row in rootChildren
        if row.title = name
            return row
        end if
    end for
end function'//# sourceMappingURL=./MainLoaderTask.bs.map