' (c) Copyright 2023, Michael Harnad, All rights reserved.
'import "pkg:/source/utils.bs"
'import "pkg:/source/Theme.bs"

'''''''''
' PlayContentWithAds: Plya video content with ads.
' 
' @param {object} I: _feedContent - the feed content to play.
' @param {string} I: _mediaType - the media type.
'''''''''
sub PlayContentWithAds(_feedContent as object, _mediaType as string) as void
    ' Set the MainScene context.
    mainScene = (function(__bsCondition, m)
            if __bsCondition then
                return m.top
            else
                return m.top.GetParent()
            end if
        end function)(m.top.SubType() = "MainScene", m)
    ' Save a reference to the Theme object.
    m.theme = Theme("MainScene")
    ' Define a bookmark object.
    m.playbackBookmark = Bookmark()
    ' Create new instance of Video object for playback.
    m.videoPlayer = mainScene.FindNode("Video")
    m.videoPlayer.visible = true
    m.videoPlayer.observeField("visible", "OnVideoPlayerVisibilityChanged")
    m.videoPlayer.observeField("state", "OnVideoPlayerStateChanged")
    ' Set some video player settings.
    m.videoPlayer.content = _feedContent
    if m.videoPlayer.hasField("playbackId")
        m.videoPlayer.SetFields({
            "playbackId": (function(__bsCondition, _feedContent)
                    if __bsCondition then
                        return _feedContent.id
                    else
                        return _feedContent.getChild(0).id
                    end if
                end function)(_mediaType <> "episode", _feedContent)
        })
    else
        m.videoPlayer.addFields({
            "playbackId": (function(__bsCondition, _feedContent)
                    if __bsCondition then
                        return _feedContent.id
                    else
                        return _feedContent.getChild(0).id
                    end if
                end function)(_mediaType <> "episode", _feedContent)
        })
    end if
    m.videoPlayer.content.addFields({
        "mediaType": _mediaType
    })
    ' Set the playback bookmark.
    m.playbackBookmark.UpdatePlaybackBookmark(m.videoPlayer)
    ShowScreen(m.videoPlayer)
    ' Set the playlist flag.
    if _mediaType = "episode"
        m.videoPlayer.contentIsPlaylist = true
    else
        m.videoPlayer.contentIsPlaylist = false
    end if
    ' Create a task to control the video playback with ads.
    m.PlayerTask = CreateObject("roSGNode", "PlayerTask")
    m.PlayerTask.observeField("state", "OnPlayerTaskStateChanged")
    m.PlayerTask.observeField("countdownSeconds", "OnCountdown")
    m.PlayerTask.video = m.videoPlayer
    m.PlayerTask.mediaType = _mediaType
    ' Start the player task.
    m.PlayerTask.control = "RUN"
end sub

'''''''''
' OnPlayerTaskStateChanged: Event handler - handles change in the task state.
' 
' @param {object} I: _state - the task state.
'''''''''
sub OnPlayerTaskStateChanged(_state as object)
    taskState = _state.GetData()
    ' Player Task is done.
    if taskState = "done"
        ' Stop watching for "state" changes.
        m.videoPlayer.UnobserveField("state")
        bookmarkToRemove = invalid
        ' Create a 'Watched' bookmark.
        if m.videoPlayer.contentIsPlaylist = false
            MarkVideoAsWatched(m.videoPlayer.content.id)
            bookmarkToRemove = m.videoPlayer.content
        else
            bookmarkToRemove = m.videoPlayer.content.getChild(0)
            MarkVideoAsWatched(m.videoPlayer.playbackId)
        end if
        ' Remove the 'Video' bookmark.
        ResetBookmark(bookmarkToRemove)
        ExitPlayback()
        m.PlayerTask.control = "stop"
        ' Player task has stopped.
    else if taskState = "stop"
        m.PlayerTask.control = "stop"
        m.videoPlayer.control = "stop"
        ExitPlayback()
    end if
end sub

'''''''''
' OnVideoPlayerVisibilityChanged: Event handler - called when the visibility of the Video node changes.
' 
'''''''''
sub OnVideoPlayerVisibilityChanged()
    ' Visibility changes when the user presses the Back key.
    ' This event is fired before the task sees it.
    if m.videoPlayer.visible = false
        ' Stop the video player.
        m.videoPlayer.control = "stop"
        m.playbackBookmark.UpdatePlaybackBookmark(m.videoPlayer)
    end if
end sub

'''''''''
' OnVideoPlayerStateChanged: Event handler - handles changes in the video player state.
' 
' @param {object} I: _state - the state of the video player.
'''''''''
sub OnVideoPlayerStateChanged(_state as object)
    state = _state.GetData()
    if state = "paused"
        m.playbackBookmark.UpdatePlaybackBookmark(m.videoPlayer)
    end if
end sub

'''''''''
' OnCountdown: Event handler - handles the countdown for the next Episode in a Series.
' 
' @param {object} I: _arg - the number of seconds left before the start of the next Episode.
'''''''''
sub OnCountdown(_arg as object)
    seconds = _arg.GetData()
    ' Check for the start trigger.
    if seconds = 0
        SetCountdownFramework()
    else if seconds = - 1
        ShowNextEpisodeCountdown(0)
    else
        ShowNextEpisodeCountdown(seconds)
    end if
end sub

'''''''''
' ExitPlayback: Exits the playback and performs cleanup.
' 
'''''''''
function ExitPlayback() as void
    CloseScreen(m.videoPlayer)
    screen = GetCurrentScreen()
    ' Return focus to the current screen.
    screen.SetFocus(true)
    ' In case of series we shouldn't change focus.
    if screen.isSeries = false AND screen.subType() = "GridScreen"
        screen.jumpToRowItem = screen.lastIndex
    end if
end function

'''''''''
' SetCountdownFramework: Set the countdown framework for next Episodes.
' 
'''''''''
function SetCountdownFramework() as void
    ' Set the MainScene context.
    mainScene = (function(__bsCondition, m)
            if __bsCondition then
                return m.top
            else
                return m.top.GetParent()
            end if
        end function)(m.top.SubType() = "MainScene", m)
    videoPlayer = m.PlayerTask.video
    ' Get the reference to some fields.
    m.nextUplabel = mainScene.FindNode("nextUpLabel")
    m.theme.SetFont(m.nextUpLabel)
    labelText = m.theme.GetObjectText("nextUpLabel", "nextUpLabel")
    m.nextUpLabel.text = m.theme.FormatText(m.nextUpLabel, labelText)
    m.nextEpisodeTitleLabel = mainScene.FindNode("nextEpisodeTitleLabel")
    m.theme.SetFont(m.nextEpisodeTitleLabel)
    m.nextEpisodeThumbnail = mainScene.FindNode("nextEpisodeThumbnail")
    m.returnImage = mainScene.FindNode("returnImage")
    m.returnImage.uri = "pkg:/images/left arrow.png"
    m.returnLabel = mainScene.FindNode("returnLabel")
    m.theme.SetFont(m.returnLabel)
    returnText = m.theme.GetObjectText("returnLabel", "returnLabel")
    m.returnLabel.text = m.theme.FormatText(m.returnLabel, returnText)
    ' Get the current Episode that is set to play.
    currentEpisode = m.global.CurrentPlaybackContent
    ' Get the next Episode.
    m.nextEpisode = FindNextEpisodeInPlaylist(currentEpisode, videoPlayer.content)
    if m.nextEpisode <> invalid
        ' Get the title of the next Episode.
        m.nextEpisodeTitleLabel.text = m.theme.FormatText(m.nextEpisodeTitleLabel, m.nextEpisode.title)
        ' Get the thumbnail for the next Episode.
        m.nextEpisodeThumbnail.uri = m.nextEpisode.thumbnail
    end if
end function

'''''''''
' ShowNextEpisodeCountdown: Show the Episode countdown.
' 
' @param {integer} I: _timeLeft - Episode time left in seconds.
'''''''''
function ShowNextEpisodeCountdown(_timeLeft as integer) as void
    if _timeleft <= 10 AND _timeleft >= 1
        if m.nextEpisode <> invalid
            labelText = m.theme.GetObjectText("nextUpLabel", "nextUpLabel")
            label = labelText + _timeleft.toStr()
            m.nextUpLabel.text = label
            m.nextUplabel.visible = true
            m.nextEpisodeTitleLabel.visible = true
            m.nextEpisodeThumbnail.visible = true
            m.returnImage.visible = true
            m.returnLabel.visible = true
        end if
    else if _timeleft = 0
        m.nextUplabel.visible = false
        m.nextEpisodeTitleLabel.visible = false
        m.nextEpisodeThumbnail.visible = false
        m.returnImage.visible = false
        m.returnLabel.visible = false
    end if
end function'//# sourceMappingURL=./VideoPlayerWithAds.bs.map