' (c) Copyright 2024, Michael Harnad



'''''''''
' SearchChannelForContent: Starts a task to perform an intra-channel search.
' 
' @param {object} I: _searchIndex - the search index.
' @param {string} I: _textInput - user keyboard input.
'''''''''
function SearchChannelForContent(_searchScreen as object, _searchIndex as object, _textInput as string) as void
    ' Set the theme for the Search Task.
    m.theme = Theme("SearchScreen")
    ' Set the font for the label.
    m.theme.SetFont(m.searchResults)
    ' Stop any previously running Search task before starting this one.
    if m.searchTask <> invalid
        m.searchTask.control = "stop"
        m.searchTask = invalid
    end if
    ' Create a Search task.
    m.searchTask = CreateObject("roSGNode", "SearchTask")
    ' Set some initial values.
    m.searchTask.searchIndex = _searchIndex
    m.searchTask.textInput = _textInput
    ' Save reference to Search screen.
    m.searchTask.searchScreen = _searchScreen
    ' Observe when the task is done.
    m.searchTask.ObserveField("searchContent", "OnSearchComplete")
    ' Check for prior Search results.  If we've already searched for this string,
    ' reload the results from the previous search.  Otherewise, start a new Search.
    if FileExists("tmp:/", "SearchResults_" + _textInput + ".json") = true
        tempFile = "tmp:/SearchResults" + "_" + _textInput + ".json"
        priorSearchResults = ParseJson(ReadAsciiFile(tempFile))
        if priorSearchResults <> invalid
            searchContent = []
            ' Extract each item in the prior Search results.
            for each item in priorSearchResults["targets"]
                savedSearchItem = m.top.callfunc("GetContentById", item.id)
                searchContent.Push(savedSearchItem)
            end for
            m.searchTask.searchContent = searchContent
        end if
    else
        ' Run the Search task.
        StartBusySpinner()
        m.searchTask.control = "run"
    end if
end function

'''''''''
' OnSearchComplete: Event handler - Handles end of search task.
'''''''''
function OnSearchComplete() as void
    ' Create the Search category buttons and add them to the Button group.
    CreateSearchCategoryButtons()
    ' Stop the Busy spinner.
    StartBusySpinner(false)
    ' Stop the Search task.
    m.searchTask.control = "stop"
    m.searchTask = invalid
    ' Get the searchResults label.
    m.searchResults = m.top.findNode("searchResults")
    m.searchResults.visible = false
end function

'''''''''
' StartBusySpinner: Starts/Stops the busy spinner.
' 
' @param {boolean} I: [_start=true] - true to start the spinner, else, false.
'''''''''
function StartBusySpinner(_start = true as boolean) as void
    if m.searchTask.muteResults <> true
        uri = ""
        control = "stop"
        if _start = true
            uri = "pkg:/images/busyspinner_hd.png"
            control = "start"
        end if
        m.spinner.uri = uri
        m.spinner.control = control
    end if
end function

'''''''''
' CreateSearchCategoryButtons: Create the Search Category Buttons.
' 
'''''''''
function CreateSearchCategoryButtons() as void
    ' Get a reference to the Search buttons.
    m.searchButtons = m.top.FindNode("searchButtons")
    ' Get the current child buttons.
    buttonChildren = m.searchButtons.getChildren(- 1, 0)
    ' Remove the prior buttons from the Group if there are any.
    if buttonChildren.Count() > 0
        m.searchButtons.removeChildren(buttonChildren)
    end if
    ' Define an array to hold individual search results.
    searchResultsArray = []
    if m.searchTask.foundMovies <> invalid AND m.searchTask.foundMovies.Count() > 0
        AddButtonToGroup("Movies", m.searchTask.foundMovies.Count())
        movies = {
            "Movies": m.searchTask.foundMovies
        }
        searchResultsArray.Push(movies)
    end if
    if m.searchTask.foundLiveFeeds <> invalid AND m.searchTask.foundLiveFeeds.Count() > 0
        AddButtonToGroup("LiveFeeds", m.searchTask.foundLiveFeeds.Count())
        livefeeds = {
            "LiveFeeds": m.searchTask.foundLiveFeeds
        }
        searchResultsArray.Push(livefeeds)
    end if
    if m.searchTask.foundSeries <> invalid AND m.searchTask.foundSeries.Count() > 0
        AddButtonToGroup("Series", m.searchTask.foundSeries.Count())
        series = {
            "Series": m.searchTask.foundSeries
        }
        searchResultsArray.Push(series)
    end if
    if m.searchTask.foundEpisodes <> invalid AND m.searchTask.foundEpisodes.Count() > 0
        AddButtonToGroup("Episodes", m.searchTask.foundEpisodes.Count())
        episodes = {
            "Episodes": m.searchTask.foundEpisodes
        }
        searchResultsArray.Push(episodes)
    end if
    if m.searchTask.foundshortformVideos <> invalid AND m.searchTask.foundshortFormVideos.Count() > 0
        AddButtonToGroup("ShortFormVideos", m.searchTask.foundshortformVideos.Count())
        shortformvideos = {
            "ShortFormVideos": m.searchTask.foundshortformVideos
        }
        searchResultsArray.Push(shortformvideos)
    end if
    if m.searchTask.foundtvSpecials <> invalid AND m.searchTask.foundtvSpecials.Count() > 0
        AddButtonToGroup("TvSpecials", m.searchTask.foundtvSpecials.Count())
        tvspecials = {
            "Tvspecials": m.searchTask.foundtvSpecials
        }
        searchResultsArray.Push(tvspecials)
    end if
    ' Save the search results to the parent Search screen.
    m.top.searchResultsArray = searchResultsArray
end function

'''''''''
' AddButtonToGroup: Adds a button to the button group.
' 
' @param {dynamic} I: _buttonName - the name of the button.
'''''''''
function AddButtonToGroup(_buttonName as string, _count as integer) as void
    button = CreateObject("roSGNode", "Button")
    text = m.theme.GetObjectText("searchButtons", _buttonName)
    button.text += text + "(" + _count.toStr() + ")"
    button.iconUri = ""
    button.focusedIconUri = ""
    button.textFont = m.searchbuttonTextfont
    button.textColor = m.buttonTextColors["textColor"]
    button.focusedTextColor = m.buttonTextColors["focusedTextColor"]
    button.focusedTextFont = m.searchbuttonFocusedTextfont
    button.height = 25.0
    button.minWidth = 50.0
    button.maxWidth = 250.0
    m.searchButtons.AppendChild(button)
end function'//# sourceMappingURL=./SearchTaskLogic.bs.map