' Bookmark class
' (c) Copyright 2023-2024, Michael Harnad
'import "pkg:/source/utils.bs"
function __Bookmark_builder()
    instance = {}
    '''''''''
    ' new: Class constructor.
    ' 
    ' @param {string} I: [section="VideoBookmarks"] - registry section name.
    '''''''''
    instance.new = sub(section = "VideoBookmarks" as string)
        m._section = invalid
        m._iRegistry = invalid
        m._iRegistrySection = invalid
        m._creationDate = invalid
        m._section = section
        ' Create the registry object.
        m._iRegistry = CreateObject("roRegistry")
        m._iRegistrySection = CreateObject("roRegistrySection", m._section)
    end sub
    ' Registry section.
    ' Registry object.
    ' Registry Section object.
    ' Creation date of the registry object.
    '''''''''
    ' GetBookmarkProperty: Gets a bookmark property from the registry entry.
    ' 
    ' @param {string} I: _contentId - the content ID key.
    ' @param {string} I: _property - the rewuested property.
    ' @return {string} - returns the property value.
    '''''''''
    instance.GetBookmarkProperty = function(_contentId as string, _property as string) as string
        regEntry = m._iRegistrySection.Read(_contentId)
        regJson = ParseJson(regEntry)
        return regJson[_property].toStr()
    end function
    '''''''''
    ' SetTimeStamp: Sets the time stamp when the registry entry is created.
    ' 
    '''''''''
    instance.SetTimeStamp = function() as void
        ' Create a DateTime object.
        date = CreateObject("roDateTime")
        m._creationDate = date.AsSeconds()
    end function
    '''''''''
    ' UpdatePlaybackPosition: Updates the video layback position.
    ' 
    ' @param {string} I: _contentId - the content ID.
    ' @param {float} I: _newPosition - the new video playback position.
    '''''''''
    instance.UpdatePlaybackPosition = function(_contentId as string, _newPosition as float) as void
        ' Init a registry entry.
        aaRegistryEntry = {}
        ' Set the creation date.
        m.SetTimeStamp()
        ' Save the playback position.
        aaRegistryEntry["position"] = _newPosition.toStr()
        ' Save the creation date.
        aaRegistryEntry["creationDate"] = m._creationDate.toStr()
        ' Create a string from the registry entry.
        regString = FormatJson(aaRegistryEntry)
        m._iRegistrySection.Write(_contentId, regString)
        m._iRegistrySection.flush()
    end function
    '''''''''
    ' UpdatePlaybackBookmark: Update the playback bookmark.
    ' 
    ' @param {object} I: _videoPlayer - the Video player object.
    '''''''''
    instance.UpdatePlaybackBookmark = function(_videoPlayer as object) as void
        ' Set the media type.
        mediaType = _videoPlayer.content.mediaType
        ' Init a registry entry.
        aaRegistryEntry = {}
        ' Series.
        if mediaType = "series"
            contentId = _videoPlayer.id
            ' Episodes.
        else if mediaType = "episode"
            if m.KeyExists(_videoPlayer.playbackId)
                aaRegistryEntry = m.ReadBookmarkEntry(_videoPlayer.playbackId)
                aaRegistryEntry["position"] = _videoPlayer.position.toStr()
                m.WriteBookmarkEntry(_videoPlayer.playbackId, aaRegistryEntry)
                return
            else
                contentId = _videoPlayer.playbackId
            end if
        else
            contentId = _videoPlayer.playbackId
        end if
        ' Save the creation date.
        if m.KeyExists(contentId) = false
            m.SetTimeStamp()
            aaRegistryEntry["creationDate"] = m._creationDate.toStr()
        else
            aaRegistryEntry["creationDate"] = m.GetBookmarkProperty(contentId, "creationDate")
        end if
        ' Save the playback position.
        aaRegistryEntry["position"] = _videoPlayer.position.toStr()
        ' Create a string from the registry entry.
        m.WriteBookmarkEntry(contentId, aaRegistryEntry)
    end function
    '''''''''
    ' WriteBookmark: Writes the bookmark to the registry.
    ' 
    ' @param {string} I: _key - the registry key.
    ' @param {string} I: value - the value to write.
    ' @return {dynamic}
    '''''''''
    instance.WriteBookmark = function(_key as string, value as string)
        m._iRegistrySection.Write(_key, value)
        m._iRegistrySection.flush()
    end function
    '''''''''
    ' ReadBookmark: Reads the bookmark from the registry.
    ' 
    ' @param {string} I: _key - the registry key.
    ' @return {string} - returns the key value.
    '''''''''
    instance.ReadBookmark = function(_key as string) as string
        return m._iRegistrySection.Read(_key)
    end function
    '''''''''
    ' ReadBookmarkEntry: Read a bookmark entry.
    ' 
    ' @param {string} I: _contentId - the content id (key).
    ' @return {object} - returns the bookmark entry (assocarray).
    '''''''''
    instance.ReadBookmarkEntry = function(_contentId as string) as object
        ' Read the registry entry.
        regEntry = m._iRegistrySection.Read(_contentId)
        ' Parse it as Json.
        regJson = ParseJson(regEntry)
        return regJson
    end function
    '''''''''
    ' WriteBookmarkEntry: Writes a bookmark entry.
    ' 
    ' @param {string} I: _contentId - the content id (key).
    ' @param {object} I: _bookmarkEntry - the bookmark entry (assocarray).
    '''''''''
    instance.WriteBookmarkEntry = function(_contentId as string, _bookmarkEntry as object) as void
        regString = FormatJson(_bookmarkEntry)
        m._iRegistrySection.Write(_contentId, regString)
        m._iRegistrySection.flush()
    end function
    '''''''''
    ' WriteEpisodeNewBookmark: Write an Episode bookmark.
    ' 
    ' @param {object} I: _episode - the Episode object.
    '''''''''
    instance.WriteNewEpisodeBookmark = function(_episode as object) as void
        ' Init a registry entry.
        aaRegistryEntry = {}
        ' Set the creation date.
        m.SetTimeStamp()
        aaRegistryEntry["creationDate"] = m._creationDate.toStr()
        ' Save the playback position.
        aaRegistryEntry["position"] = 0
        ' Write the bookmark.
        m.WriteBookmarkEntry(_episode.id, aaRegistryEntry)
    end function
    '''''''''
    ' DeleteTokenBookmark: Deletes the token bookmark from the registry.
    ' 
    ' @param {string} I: _key - the registry key.
    ' @return {dynamic}
    '''''''''
    instance.DeleteTokenBookmark = function(_key as string)
        m._iRegistrySection.Delete(_key)
        m._iRegistrySection.flush()
    end function
    '''''''''
    ' KeyExists: Checks for the existence of the regisry key.
    ' 
    ' @param {string} I: _key - the key to check.
    ' @return {boolean} - returns true if found, else, false.
    '''''''''
    instance.KeyExists = function(_key as string) as boolean
        ' Set a result flag.
        found = false
        ' Get the registry key list of content ids.
        keyList = m.GetKeyList()
        for each key in keyList
            if key = _key
                found = true
                exit for
            end if
        end for
        return found
    end function
    '''''''''
    ' GetKeyList: Gets the list of registry keys for the section.
    ' 
    ' @return {object} - returns a list of the registry keys.
    '''''''''
    instance.GetKeyList = function() as object
        return m._iRegistrySection.GetKeyList()
    end function
    '''''''''
    ' GetVideoPlaybackPercentage: Gets the video playback percentage.
    ' 
    ' @param {string} I: _contentId - the content id.
    ' @param {integer} I: videoDuration - the video duration.
    ' @param {float} I: [newPosition=invalid] - the new video position.
    ' @return {float} Returns the total video playback percentage.
    '''''''''
    instance.GetVideoPlaybackPercentage = function(_contentId as string, videoDuration as integer, newPosition = invalid as float) as float
        if newPosition = invalid
            lastPosition = m.GetLastVideoPosition(_contentId)
        else
            lastPosition = newPosition
        end if
        try
            percentage = lastPosition / videoDuration
        catch excp
            percentage = 0
        end try
        return percentage
    end function
    '''''''''
    ' UpdateVideoPosition: Updates the video postion in the bookmark.
    ' 
    ' @param {string} I: _contentId - the content Id.
    ' @param {float} I: newPosition - the new video content position.
    ' @param {integer} I: videoDuration - the duration of the video content.
    ' @return {dynamic}
    '''''''''
    instance.UpdateVideoPosition = function(_contentId as string, newPosition as float, videoDuration as integer)
        percentage = m.GetVideoPlaybackPercentage(_contentId, videoDuration, newPosition)
        ' If the playback postion is at least 95% complete, delete the bookmark.
        if percentage > 0.95
            m.DeleteBookmark(_contentId)
        else
            m.UpdatePlaybackPosition(_contentId, newPosition)
        end if
    end function
    '''''''''
    ' GetLastVideoPosition: Get the last video position.
    ' 
    ' @param {string} _contentId - the id of the content.
    ' @return {float} the last video position.
    '''''''''
    instance.GetLastVideoPosition = function(_contentId as string) as float
        if m._iRegistrySection.Exists(_contentId)
            strPosition = m.GetBookmarkProperty(_contentId, "position")
            return strPosition.toFloat()
        else
            return 0.0
        end if
    end function
    '''''''''
    ' VideoCanBeResumed: Checks to see if the video can be resumed.
    ' 
    ' @param {string} I: _contentId - id of video to check.
    ' @param {object} O: positiion - current video position.
    ' @return {boolean} returns true if video can be resumed, else, false.
    '''''''''
    instance.VideoCanBeResumed = function(_contentId as string, position as object) as boolean
        if m._iRegistrySection.Exists(_contentId)
            strPosition = m.GetBookmarkProperty(_contentId, "position")
            if strPosition.toFloat() > 0
                position[0] = strPosition.toFloat()
                return true
            else
                position[0] = 0.0
                return false
            end if
        else
            position[0] = 0.0
            return false
        end if
    end function
    '''''''''
    ' DeleteBookmark: Delete the content bookmark.
    ' 
    ' @param {string} _contentId - id of the content.
    ' @return {dynamic}
    '''''''''
    instance.DeleteBookmark = function(_contentId as string)
        if m._iRegistrySection.Exists(_contentId)
            m._iRegistrySection.Delete(_contentId)
            m._iRegistrySection.flush()
        end if
    end function
    '''''''''
    ' GetRegistrySectionObject: Gets the registry section object by name.
    ' 
    ' @return {object} returns the registry Section object.
    '''''''''
    instance.GetRegistrySectionObject = function() as object
        return m._iRegistrySection
    end function
    '''''''''
    ' DebugListRegistrySections: Iterates the registry sections to the debug window.
    ' 
    ' @return {dynamic}
    '''''''''
    instance.DebugListRegistrySections = function()
        for each sectionItem in m.iRegistry.GetsectionList()
            ? ">> DEBUG Section name= " sectionItem
            registrySection = m.GetRegistrySectionObject()
            sectionKeys = registrySection.GetKeylist()
            for each sectionKey in sectionKeys
                value = registrySection.Read(sectionKey)
                ? ">> DEBUG content id= " sectionKey " position= " value
            end for
        end for
    end function
    return instance
end function
function Bookmark(section = "VideoBookmarks" as string)
    instance = __Bookmark_builder()
    instance.new(section)
    return instance
end function'//# sourceMappingURL=./BookmarkLogic.bs.map