' (c) Copyright 2024, Michael Harnad
'import "pkg:/source/utils.bs"
'import "pkg:/components/UILogic/VideoPlayer.bs"


sub Init()
    m.top.tagSelectedContent = []
    ' Watch for the tag grid content.
    m.top.observeField("tagSelectedContent", "OnContentProvided")
    m.top.observeField("visible", "OnVisibleChanged")
    ' Get the reference to the main menu.
    m.mainMenu = m.top.FindNode("mainMenu")
    m.menuList = m.mainMenu.getChild(0)
    ' Get a reference to the poster grid.
    m.tagContentGrid = m.top.FindNode("TagContentGrid")
    m.tagContentGrid.observeField("itemSelected", "OnContentTagSelected")
end sub

'''''''''
' OnContentProvided: Event handler - called when the tag content has been provided.
' 
'''''''''
sub OnContentProvided()
    ' Build the tag content grid.
    BuildTagContentGrid()
    m.menuList.drawFocusFeedback = false
    m.tagContentGrid.SetFocus(true)
end sub

'''''''''
' OnContentTagSelected: Event handler - handles selection of an item in the grid.
' 
' @param {object} _tagIndex
'''''''''
sub OnContentTagSelected(_tagIndex as object)
    ' Define a AA array to create the ContentNode.
    args = {}
    ' Get the selected content index.
    tagIndex = _tagIndex.GetData()
    ' Get the associated tag content.
    tagContent = m.tagContentGrid.content.getChild(tagIndex)
    args["dl_mediaType"] = tagContent.mediaType
    args["dl_id"] = tagContent.id
    ' Create a ContentNode for the selected content tag.
    content = CreateContentNodeFromRawObject(args)
    flag = bslib_ternary(args["dl_mediaType"] = "series", true, false)
    ShowVideoScreen(content, 0, flag)
end sub

'''''''''
' OnVisibleChanged: Event handler - handles changes in screen visibility.
' 
' @param {object} I: _visibility - true if visible, else, false.
'''''''''
sub OnVisibleChanged(_visibility as object)
    isVisible = _visibility.GetData()
    if isVisible
        m.tagContentGrid.SetFocus(true)
    end if
end sub

'''''''''
' CalculateEpisodeContentDuration: Calculate the total duration of Episodes.
' 
' @param {object} I: _Series - the raw Series object.
' @return {string} - returns the total duraton as HH MM SS.
'''''''''
function CalculateEpisodeContentDuration(_Series as object) as string
    duration = 0
    if _Series.seasons <> invalid and _Series.seasons.Count() > 0
        for each season in _Series.seasons
            for each episode in season.episodes
                duration += episode.content.duration
            end for
        end for
    else
        for each episode in _Series.episodes
            duration += episode.content.duration
        end for
    end if
    return GetTime(duration)
end function

'''''''''
' BuildTagContentGrid: Builds the tag content grid.
' 
'''''''''
function BuildTagContentGrid() as void
    ' Create the tag content root node.
    tagContentRoot = CreateObject("roSGNode", "ContentNode")
    ' Iterate to append the selected tag items.
    for each selectedTagItem in m.top.tagSelectedContent
        tagContent = CreateObject("roSGNode", "ContentNode")
        contentObject = ExtractRawContentFromFeed(selectedTagItem["mediaType"], selectedTagItem["id"])
        description = contentObject.title
        if not selectedTagItem["mediaType"] = "series"
            description += " (" + GetTime(contentObject.content.duration) + ")"
        else
            description += " (" + CalculateEpisodeContentDuration(contentObject) + ")"
        end if
        tagContent.shortdescriptionline1 = description
        tagContent.HDPosterUrl = contentObject.thumbnail
        tagContent.SetFields({
            "id": contentObject.id
        })
        tagContent.addFields({
            "mediaType": selectedTagItem["mediaType"]
        })
        tagContentRoot.appendChild(tagContent)
    end for
    m.tagContentGrid.content = tagContentRoot
end function

'''''''''
' onKeyEvent: Event Handler - handles remote key presses.
' 
' @param {string} I: _key - key that was pressed.
' @param {boolean} I: _press - true if key was pressed, false if released.
' @return {boolean} - returns true if handled, else, false.
'''''''''
function onKeyEvent(_key as string, _press as boolean) as boolean
    handled = false
    ' If Left key press and the grid has focus...
    if _key = "left" and _press = true and m.tagContentGrid.hasFocus()
        m.menuList.itemSize = [
            150
            35
        ]
        m.menuList.jumpToItem = m.menuList.itemFocused
        m.menuList.drawFocusFeedback = true
        m.menuList.SetFocus(true)
        handled = true
    end if
    return handled
end function'//# sourceMappingURL=./TagContentScreen.bs.map