'import "pkg:/components/UILogic/SearchScreenLogic.bs"
'import "pkg:/components/UILogic/DetailsScreenLogic.bs"
'import "pkg:/components/UILogic/LoginStatusLogic.bs"
'import "pkg:/components/UILogic/BookmarkLogic.bs"
'import "pkg:/components/UILogic/AccessTokenLogic.bs"
'import "pkg:/components/UILogic/ScreenStackLogic.bs"
'import "pkg:/components/UILogic/SearchTaskLogic.bs"
'import "pkg:/source/utils.bs"
'import "pkg:/source/Theme.bs"
' Start the Search after this many seconds have elapsed without user key input.


sub Init()
    ' Set the theme for the Search Screen.
    m.theme = Theme("SearchScreen")
    ' observe "visible" so we can know when DetailsScreen change visibility
    m.top.ObserveField("visible", "OnVisibleChange")
    ' Save the searchList object.
    m.searchList = m.top.findNode("searchList")
    m.searchList.drawFocusFeedback = false
    m.searchList.ObserveField("itemSelected", "OnSearchItemSelected")
    ' Save the searchResults label.
    m.searchResults = m.top.findNode("searchResults")
    ' Set the font for the label.
    m.theme.SetFont(m.searchResults)
    ' Get the initial label text.
    searchResults = m.theme.GetObjectText("searchResults", "noResults")
    m.searchResults.text = m.theme.FormatText(m.searchResults, searchResults)
    ' Observe 'text' so we know when the user enters some text.
    m.miniKeyboard = m.top.findNode("searchKeyboard")
    m.miniKeyboard.ObserveField("text", "OnTextChanged")
    m.miniKeyboard.setFocus(true)
    ' Display 'busy' animation.
    m.spinner = m.top.findNode("busySpinner")
    m.spinner.poster.observeField("loadStatus", "OnShowSpinner")
    m.spinner.control = "stop"
    ' Define a timer to detect user keystroke end.
    m.startSearchTimer = m.top.FindNode("startTimer")
    m.startSearchTimer.duration = 2.5
    m.startSearchTimer.ObserveField("fire", "OnStartSearch")
    ' Get a reference to the Search category buttons.
    m.searchButtons = m.top.FindNode("searchButtons")
    m.buttonTextColors = {}
    ' Define an array to hold the button fonts.
    SetButtonFont()
end sub

'''''''''
' SetButtonFont: Set the font for the Search button group.
'
'''''''''
function SetButtonFont() as void
    m.searchbuttonTextfont = m.theme.SetButtonFont(m.searchButtons, m.buttonTextColors)
    m.searchButtons.textFont = m.searchbuttonTextfont
    m.searchbuttonFocusedTextfont = m.theme.SetButtonFont(m.searchButtons, m.buttonTextColors)
    m.searchButtons.focusedTextFont = m.searchbuttonFocusedTextfont
end function

'''''''''
' OnShowSpinner: Event handler - process spinner load status.
' 
'''''''''
sub OnShowSpinner()
    if m.spinner.poster.loadStatus = "ready"
        centerx = (1280 - m.spinner.poster.bitmapWidth) / 2
        centery = (720 - m.spinner.poster.bitmapHeight) / 2
        m.spinner.translation = [
            centerx
            centery
        ]
        m.spinner.visible = "true"
    end if
end sub

sub OnVisibleChange() ' invoked when SearchScreen visibility is changed
    if m.top.visible = true
        ' Focus the last selected search item.
        m.searchList.jumpToItem = m.itemSelected
        m.searchList.setFocus(true)
    end if
end sub

'''''''''
' OnSearchItemSelected: Event handler - handles search item selection.
' 
' @param {object} event
'''''''''
sub OnSearchItemSelected(event as Object) ' invoked when search item is selected
    ' Get the index of the selected item.
    m.itemSelected = event.GetData()
    ' Create a copy of the selected content as a ContentNode so that it can be played.
    searchListItem = m.searchList.content.getChild(m.itemSelected).Clone(true)
    HandleSearchSelection(searchListItem)
end sub

'''''''''
' OnTextChanged: Event handler - Handles text change event for the mini keyboard.
' 
'''''''''
function OnTextChanged() as void
    ' Get the text input.
    textInput = m.miniKeyboard.text
    if Len(textInput) = 0
        m.searchList.content = invalid
        m.searchButtons.visible = false
        searchResults = m.theme.GetObjectText("searchResults", "noResults")
        UpdateResultsLabel(m.theme.FormatText(m.searchResults, searchResults))
    else
        ' Stop the search timer.
        m.startSearchTimer.control = "stop"
        ' Restart the search timer.
        m.startSearchTimer.control = "start"
    end if
end function

'''''''''
' OnStartSearch: Event handler - Handles start of a Search process triggered by the timer.
' 
'''''''''
function OnStartSearch() as void
    ' Get the text input.
    textInput = m.miniKeyboard.text
    UpdateResultsLabel(m.theme.FormatText(m.searchResults, "Searching...please wait"))
    SearchChannelForContent(m.top, m.global.ContentIndex, textInput)
end function

'''''''''
' UpdateResultsLabel: Update the Search results label text.
' 
' @param {string} I: _string - the string to update the results label.
' @param {boolean} I: [_show=true] - true to show the label, else, false.
'''''''''
function UpdateResultsLabel(_string as string, _show = true as boolean) as void
    m.searchResults.text = _string
    m.searchResults.visible = _show
end function

'''''''''
' OnKeyEvent: Event Handler: Handles remote key presses.
' 
' @param {string} I: key - the key that as pressed.
' @param {boolean} I: press - flag to indicate a key press.
' @return {boolean} - returns true if this code handles the key press.
'''''''''
function OnKeyEvent(key as String, press as Boolean) as Boolean
    handled = false
    if press = true
        ' RIGHT key press.
        if key = "right"
            m.searchList.SetFocus(true)
            m.searchList.drawFocusFeedback = true
            handled = true
        end if
        ' OPTIONS key press.
        if m.searchList.hasFocus() = true and key = "options"
            message = []
            searchContent = m.searchList.content.getChild(m.searchList.itemFocused)
            message.push(searchContent.longDescription)
            m.descriptionDialog = CreateMessageDialog(searchContent.title, message, [
                "Ok"
            ])
            m.descriptionDialog.palette = CreateADialogPalette()
            ' observe the dialog's buttonSelected field to handle button selections
            m.descriptionDialog.observeFieldScoped("buttonSelected", "onOptionOkButtonSelected")
            ' display the dialog
            scene = m.top.getScene()
            scene.dialog = m.descriptionDialog
            handled = true
        end if
    end if
    return handled
end function

'''''''''
' ReloadSearchResults: Reloads the Search results list with content specific results.
' 
' @param {integer} I: _index - index of the content in the Search array.
'''''''''
function ReloadSearchResults(_index as integer) as void
    searchResults = m.searchList.content.getChildren(- 1, 0)
    m.searchList.content.removeChildren(searchResults)
    contentSearchResults = m.top.searchResultsArray[_index]
    for each item in contentSearchResults.Items()
        m.searchList.content.appendChildren(item.value)
    end for
end function

'''''''''
' onOptionOkButtonSelected: Event handler - Handles OK button press for Options dialog.
' 
'''''''''
sub onOptionOkButtonSelected()
    if m.descriptionDialog.buttonSelected = 0
        ' handle ok button here
        m.descriptionDialog.close = true
    end if
end sub

'''''''''
' HandleSearchSelection: Handles the selection of a Search item.
' 
' @param {object} I: _searchListItem - the Search list item to play.
'''''''''
sub HandleSearchSelection(_searchListItem as Object)
    ' Save the current playback content to the global node.
    m.global.addFields({
        "CurrentPlaybackContent": _searchListItem
    })
    m.global.addFields({
        "searchId": _searchListItem.id
    })
    seriesFlag = false
    ' If search content is a Series, default to play all.
    if _searchListItem.mediaType = "series"
        _searchListItem.addFields({
            "PlayAll": true
            "SelectedSeasonIndex": 0
            "SelectedEpisodeIndex": 0
        })
        seriesFlag = true
        AppendEpisodesToSeriesNode(_searchListItem, _searchListItem.hasSeasons)
    end if
    if not IsAudioContent(_searchListItem)
        ' Play Video content.
        if m.global.useSubscriptions <> true
            ShowVideoScreen(_searchListItem, 0, seriesFlag)
        else
            CheckSubscriptionAndStartPlayback(_searchListItem, 0, seriesFlag)
        end if
    else
        ' Play Audio content.
        ShowAudioScreen(_searchListItem.url, _searchListItem.HDPosterUrl, _searchListItem.shortDescription)
    end if
end sub

'''''''''
' GetContentById: Gets content node based on the content ID.
' 
' @param {string} I: _id - content id.
' @return {object} - returns the content object.
'''''''''
function GetContentById(_id as string) as object
    node = invalid
    contentRows = m.top.content.getChildren(- 1, 0)
    for each rowItem in contentRows
        node = FindNodeById(rowItem, _id)
        if node <> invalid
            exit for
        end if
    end for
    return node
end function'//# sourceMappingURL=./SearchScreen.bs.map