' (c) Copyright 2024, Michael Harnad
'import "pkg:/source/Theme.bs"



' entry point of GridScreen.
' Note that we need to iport this file in GridScreen.xml using relative path.
sub Init()
    ' Get the reference to the main menu.
    m.mainMenu = m.top.FindNode("mainMenu")
    m.menuList = m.mainMenu.getChild(0)
    ' Save a reference to the Scene.
    m.mainScene = m.top.getScene()
    ' Save a reference to the Theme object.
    m.theme = Theme(m.top.SubType())
    ' Save a reference to the Video bookmarks.
    m.videoBookmark = Bookmark()
    ApplyOverhangOptions()
    m.rowList = m.top.FindNode("rowList")
    m.rowList.SetFocus(true)
    ' observe visible field so we can see GridScreen change visibility.
    m.top.ObserveField("visible", "OnVisibleChange")
    ' label with item title.
    m.titleLabel = m.top.FindNode("titleLabel")
    m.theme.setFont(m.titleLabel)
    ' label with item description.
    m.descriptionLabel = m.top.FindNode("descriptionLabel")
    m.theme.setFont(m.descriptionLabel)
    ' content thumbnail.
    m.contentThumbnail = m.top.FindNode("contentThumbnail")
    ' extras label.
    m.extrasLabel = m.top.FindNode("extrasLabel")
    m.theme.setFont(m.extrasLabel)
    ' observe rowItemFocused so we can know when another item of rowList will be focused.
    m.rowList.ObserveField("rowItemFocused", "OnItemFocused")
end sub

sub OnVisibleChange()
    if m.top.visible = true
        ' Update the Continue watching row if necessary.
        UpdateContinueWatchingRow()
        m.rowList.SetFocus(true) ' set focus to RowList if GridScreen is visible.
    end if
end sub

sub OnItemFocused() ' invoked when another item is focused.
    focusedIndex = m.rowList.rowItemFocused ' get position of focused item.
    row = m.rowList.content.GetChild(focusedIndex[0]) ' get all items of row.
    item = row.GetChild(focusedIndex[1]) ' get focused item.
    if item <> invalid
        ApplyManifestOptions(item)
        ' update title label with title of focused item. 
        m.titleLabel.text = m.theme.FormatText(m.titleLabel, item.title)
        ' update description label with description of focused item.
        m.descriptionLabel.text = m.theme.FormatText(m.descriptionLabel, item.shortDescription)
        ' Add genres, release date, etc.
        m.extrasLabel.text = m.theme.FormatText(m.extrasLabel, Left(item.releaseDate, 4))
        m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, chr(62504), 1)
        if item.genres <> invalid
            for each genre in item.genres
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, genre)
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, chr(62504), 1)
            end for
        end if
        if item.mediaType <> "series"
            if item.ratingObj <> invalid AND item.ratingObj["rating"] <> ""
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, item.ratingObj["rating"])
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, chr(62504), 1)
            end if
        end if
        ' If the media type is NOT a "series", show the duration.  Otherwise, show the number of Seasons/Episodes.
        if item.mediaType <> "series"
            if item.length <> invalid and item.length <> 0
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, GetTime(item.length))
            end if
        else
            if item.hasSeasons = true
                m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, item.numSeasons.toStr() + " Seasons ")
            end if
            m.extrasLabel.text += m.theme.FormatText(m.extrasLabel, item.numEpisodes.toStr() + " Episodes")
        end if
        if item.mediaType <> "episode"
            m.contentThumbnail.uri = item.hdposterurl
        else
            ' Use the Series thumbnail for an associated Episode.
            rawSeries = GetSeriesParentObjectFromEpisode(item)
            m.contentThumbnail.uri = rawSeries.thumbnail
        end if
    else
        m.titleLabel.text = "WARNING: Missing content."
        m.descriptionLabel.text = "Content for this category has not been defined."
        m.extrasLabel.text = ""
        m.contentThumbnail.uri = ""
    end if
end sub

'''''''''
' ApplyManifestOptions: Apply any manifest options.
' 
' @param {object} I: _item - the selected grid item.
' @return {dynamic}
'''''''''
function ApplyManifestOptions(_item as object)
    ' Check for dynamic background assignment.
    if m.global.USE_THUMBNAIL_BACKGROUND and _item <> invalid
        m.mainScene.backgrounduri = _item.hdposterurl
    end if
end function

'''''''''
' ApplyOverhangOptions: Applies options to the overhang component.
' 
' @return {dynamic}
'''''''''
function ApplyOverhangOptions()
    overhang = m.top.FindNode("overhang")
    hideGridThumbnail = m.global.HIDE_GRID_THUMBNAIL
    if hideGridThumbnail
        ' Hide the thumbnail poster.
        m.top.FindNode("contentThumbnail").visible = false
        ' Expand the overhang scale if necessary.
        if overhang <> invalid
            overhang.scale = "[1.0, 1.0]"
        end if
    end if
    if overhang <> invalid
        if not m.global.OVERHANG_CLOCK
            overhang.showClock = false
        end if
    end if
end function

'''''''''
' UpdateContinueWatchingRow: Updates the 'Continue watching' row.
' 
'''''''''
function UpdateContinueWatchingRow() as void
    ' Get the grid children and the first row.
    gridChildren = m.top.content.getChildren(- 1, 0)
    ' Get a reference to the Grid row children.
    gridRows = []
    gridRows.Append(gridChildren)
    ' Get the first row in the grid.
    firstGridRow = gridChildren[0]
    ' Set a flag indicating whether the 'Continue watching' row is present.
    continueWatchingRow = bslib_ternary(firstGridRow.title = "Continue watching", true, false)
    ' Initialize selected Grid content from the global node.
    selectedGridContent = m.global.CurrentPlaybackContent
    ' Check for a Series/Episode playback.
    if selectedGridContent <> invalid AND ContentIsType(selectedGridContent, "series")
        if SeriesHasSeasons(selectedGridContent) = false
            selectedGridContent = GetEpisodeFromSeries(selectedGridContent, selectedGridContent.selectedEpisodeIndex)
        else
            Season = GetSeasonFromSeries(selectedGridContent, selectedGridContent.selectedseasonIndex)
            selectedGridContent = GetEpisodeFromSeason(Season, selectedGridContent.selectedEpisodeIndex)
        end if
        'else if selectedGridContent <> invalid AND ContentIsType(selectedGridContent, "episode")
    end if
    ' Ensure the selected content is valid before proceeding.
    if selectedGridContent = invalid
        return
    end if
    ' Get the list of bookmark content keys.  The keys are
    ' the id's of all content that can be continued.
    regKeyList = m.videoBookmark.GetKeyList()
    if regKeyList.Count() > 0
        for each key in regKeyList
            ' Can the currently selected item be continued?
            if key = selectedGridContent.id
                if continueWatchingRow = false
                    ' Build a 'Continue watching' row for the grid.
                    continueNode = CreateObject("roSGNode", "ContentNode")
                    continueNode.Update({
                        "order": "most_recent"
                        "title": "Continue watching"
                    }, true)
                    ' Add the selected item as a clone to the 'Continue watching' row.
                    continueNode.AppendChild(selectedGridContent.clone(true))
                    gridRows.Unshift(continueNode)
                else
                    addItemToContinueWatching = true
                    ' Scan the first row to make sure the selected item is not already in the row.
                    for each child in firstGridRow.getChildren(- 1, 0)
                        if child.id = selectedGridContent.id
                            addItemToContinueWatching = false
                            exit for
                        end if
                    end for
                    if addItemToContinueWatching = true
                        ' Add the selected item as a clone to the 'Continue watching' row.
                        firstGridRow.AppendChild(selectedGridContent.clone(true))
                    end if
                end if
                exit for
            end if
        end for
        ' Get a Watched bookmark.
        watchedBM = Bookmark("Watched")
        ' Get the Watched keys.
        watchedKeyList = watchedBM.GetKeyList()
        ' Get the children of the Continue watching row.
        continueWatchingChildren = firstGridRow.getChildren(- 1, 0)
        ' If the Watched keylist contains items, scan them...
        if watchedKeyList.Count() > 0
            for each key in watchedKeyList
                index = 0
                ' Scan the Continue Watching row...
                for each rowzeroChild in continueWatchingChildren
                    ' If a key from the Watched list matches an ID in Continue Watching, remove it.
                    if key = rowzeroChild.id
                        firstGridRow.removeChildIndex(index)
                        gridRows[0] = firstGridRow
                        exit for
                    end if
                    index++
                end for
            end for
        end if
        m.rowList.content.Update({
            children: gridRows
        }, true)
        ' If the Continue watching row is present, remove it since there is
        ' no content to continue.    
    else if continueWatchingRow = true
        m.rowList.content.removeChildIndex(0)
    end if
end function

'''''''''
' onKeyEvent: Event handler - handles remote key press events.
' 
' @param {string} I: _key - the key string.
' @param {boolean} I: _press - true if pressed, else, false if released.
' @return {boolean} - returns true to handle the event, else, false.
'''''''''
function onKeyEvent(_key as string, _press as boolean) as boolean
    result = false
    ' Check for LEFT key press.
    if _key = "left" and _press = true
        ' Slide the screen to the Right.
        m.menuList.itemSize = [
            150
            35
        ]
        m.menuList.jumpToItem = m.menuList.itemFocused
        m.menuList.drawFocusFeedback = true
        m.menuList.SetFocus(true)
        result = true
    end if
    return result
end function'//# sourceMappingURL=./GridScreen.bs.map