' entry point of EpisodesScreen
'import "pkg:/source/Theme.bs"

function Init()
    ' Save a reference to the Theme object.
    m.theme = Theme(m.top.SubType())
    m.seasonsList = m.top.FindNode("seasonsList")
    ' observe "visible" so we can know when EpisodesScreen change visibility
    m.top.ObserveField("visible", "onVisibleChange")
    ' observe "itemFocused" so we can know which season gain focus
    m.seasonsList.ObserveField("itemFocused", "OnSeasonsItemFocused")
    m.seasonIndex = 0
    m.episodesLabel = m.top.FindNode("episodesLabel")
    m.theme.SetFont(m.episodesLabel)
    m.episodesList = m.top.FindNode("episodesList")
    ' observe "itemFocused" so we can know which episode gain focus
    m.episodesList.ObserveField("itemFocused", "OnEpisodeListItemFocused")
    ' observe "itemSelected" so we can know which episode is selected
    m.episodesList.ObserveField("itemSelected", "OnListItemSelected")
    m.top.ObserveField("content", "OnContentChange")
    m.episodeIndex = 0
    ' voting label.
    m.votingLabel = m.top.FindNode("votingLabel")
    m.theme.SetFont(m.votingLabel)
    m.votingLabel.visible = false
    ' voting timer.
    m.votingTimer = m.top.FindNode("votingTimer")
    ' Get the Content Voting flag.
    m.contentVoting = m.global.CONTENT_VOTING
end function

sub OnEpisodeListItemFocused(event as Object) ' invoked when episode is focused with the remote.
    focusedItem = event.GetData() ' index of episode
    m.episodeIndex = focusedItem
    m.episodesLabel.visible = true
    epNumber = focusedItem + 1
    ' Build a fake 'tab' stream.
    tabChr = ""
    for i = 0 to 75
        tabChr += Chr(32)
    end for
    m.episodesLabel.text = m.theme.FormatText(m.episodesLabel, tabChr + epNumber.toStr() + " of " + m.episodesList.content.getChildCount().toStr())
    UnformatSeasonTitle(m.seasonsList.itemFocused)
    ' index of season which contains focused episode
    if (m.seasonIndex - 1) = m.seasonsList.jumpToItem
        m.seasonsList.animateToItem = m.seasonIndex
    else if not m.seasonsList.IsInFocusChain()
        m.seasonsList.jumpToItem = m.seasonIndex
    end if
end sub

'''''''''
' FillEpisodeList: Sets the items in the Episodes list.
' 
' @param {object} I: _season - the selected Season object.
' @return {integer} - returns the number of Episodes for the Season.
'''''''''
function FillEpisodeList(_season as object) as integer
    episodes = []
    for each episode in _season.GetChildren(- 1, 0)
        fields = episode.getFields()
        episodes.Push(fields)
    end for
    ' Populate episodesList with list of Episodes.
    m.episodesList.content = EpisodeContentListToSimpleNode(episodes)
    return episodes.Count()
end function

'''''''''
' FillSeasonList: Fills the Season list.
' 
' @param {object} I: _content - the Series content.
'''''''''
sub FillSeasonList(_content as Object)
    ' save the title of each season
    seasons = []
    for each season in _content.GetChildren(- 1, 0)
        seasons.push({
            title: season.title
        }) ' save title of each season
    end for
    m.seasonsList.content = ContentListToSimpleNode(seasons) ' populate seasonsList with list of seasons
end sub

sub OnSeasonsItemFocused(event as Object) ' invoked when Season is focused
    ' Get the Season index.
    focusedItem = event.GetData()
    m.seasonIndex = focusedItem
    m.episodeIndex = 0
    m.episodesLabel.visible = false
    season = m.top.content.getChild(m.seasonIndex)
    numEpisodes = FillEpisodeList(season)
    FormatSeasonTitles(numEpisodes)
    ' we shouldn't change the focus in the episodes list as soon as we have switched to the list of seasons
    if m.seasonsListGainFocus = true
        m.seasonsListGainFocus = false
    else
        ' navigate to the first episode of season
        m.episodesList.jumpToItem = m.episodeIndex
    end if
end sub

sub OnJumpToItem() ' invoked when "jumpToItem field is changed
    m.episodesList.jumpToItem = m.episodeIndex
end sub

sub OnContentChange() ' invoked when EpisodesScreen content is changed
    content = m.top.content
    FillSeasonList(content) ' populate seasons list
    season = content.GetChild(m.seasonIndex)
    FillEpisodeList(season)
end sub

'''''''''
' onVisibleChange: Event Handler - handles the screen visibility change.
' 
'''''''''
sub onVisibleChange()
    if m.top.visible = true
        m.episodesList.jumpToItem = m.top.selectedItem[1]
        ' Set the focus to the Episodes list.
        m.episodesList.setFocus(true)
    end if
end sub

sub OnListItemSelected() ' invoked when Episode is selected
    m.top.selectedItem = [
        m.seasonIndex
        m.episodeIndex
    ]
end sub

' The OnKeyEvent() function receives remote control key events
function OnKeyEvent(key as String, press as Boolean) as Boolean
    result = false
    if press
        ' Look for Options press to display full text.
        if key = "options"
            episodes = m.episodesList.content.GetChildren(- 1, 0)
            episode = episodes[m.episodeIndex]
            message = []
            if episode.Actors <> invalid and episode.Actors.Count() > 0
                for each actor in episode.Actors
                    message.Push("Actor: " + actor)
                end for
            end if
            if episode.Hosts <> invalid and episode.Hosts.Count() > 0
                for each host in episode.Hosts
                    message.Push("Host: " + host)
                end for
            end if
            if episode.Narrators <> invalid and episode.Narrators.Count() > 0
                for each narrator in episode.Narrators
                    message.Push("Narrator: " + narrator)
                end for
            end if
            if episode.Directors <> invalid and episode.Directors.Count() > 0
                for each director in episode.Directors
                    message.Push("Director: " + director)
                end for
            end if
            if episode.Producers <> invalid and episode.Producers.Count() > 0
                for each producer in episode.Producers
                    message.Push("Producer: " + producer)
                end for
            end if
            ' Check for a long description first.  Otherwise, use the short description.
            if episode.longDescription <> invalid
                message.Push(episode.longDescription)
            else if episode.shortDescription <> invalid
                message.Push(episode.shortDescription)
            end if
            ' Set the dialog button(s).
            if m.contentVoting = invalid OR m.contentVoting = false
                m.descriptionDialog = CreateMessageDialog(episode.title, message, [
                    "Ok"
                ])
            else
                m.descriptionDialog = CreateMessageDialog(episode.title, message, [
                    "Like"
                    "Dislike"
                ])
            end if
            m.descriptionDialog.palette = CreateADialogPalette()
            ' observe the dialog's buttonSelected field to handle button selections
            m.descriptionDialog.observeFieldScoped("buttonSelected", "onOptionVoteButtonSelected")
            ' display the dialog
            scene = m.top.getScene()
            scene.dialog = m.descriptionDialog
            result = true
            ' handle "left" key press
        else if key = "left" and m.episodesList.HasFocus() ' episodes list should be focused
            m.seasonsListGainFocus = true
            ' navigate to seasons list
            m.seasonsList.SetFocus(true)
            m.episodesList.drawFocusFeedback = false
            result = true
            ' handle "right" key press
        else if key = "right" and m.seasonsList.HasFocus() ' seasons list should be focused
            m.episodesList.drawFocusFeedback = true
            ' navigate to episodes list
            m.episodesList.SetFocus(true)
            result = true
            ' Look for 'replay'/'restart' key.
        else if key = "replay"
            ' Get the currently focused Episode.
            episode = m.episodesList.content.getChild(m.episodesList.itemFocused)
            ' Set the restart flag.
            episode.addFields({
                "restart": true
            })
            ' Remove the Episode bookmark, so that the playback restarts.
            episodeBookmark = Bookmark()
            episodeBookmark.DeleteBookmark(episode.id)
            ' Trigger the playback.
            m.top.selectedItem = [
                m.seasonIndex
                m.episodeIndex
            ]
            result = true
        end if
    end if
    return result
end function

'''''''''
' FormatSeasonTitles: Formats the currnt selected Season title.
' 
' @param {integer} I: _numEpisodes - number of Episodes for the Season.
' @return {dynamic}
'''''''''
function FormatSeasonTitles(_numEpisodes as integer)
    seasonCurrentContentNode = m.seasonsList.content.GetChild(m.seasonsList.itemFocused)
    seasonCurrentContentNode.title += "     " + _numEpisodes.toStr() + " Episodes"
    seasonPrevContentNode = m.seasonsList.content.GetChild(m.seasonsList.itemUnfocused)
    if seasonPrevContentNode <> invalid
        regxp = CreateObject("roRegex", "\b     \b", "i")
        seasonTitle = regxp.Split(seasonPrevContentNode.title)
        seasonPrevContentNode.title = seasonTitle[0]
    end if
end function

'''''''''
' UnformatSeasonTitle: Unformats the current selected Season title.
' 
' @param {integer} I: _seasonIndex - index of the current Season.
' @return {dynamic}
'''''''''
function UnformatSeasonTitle(_seasonIndex as integer)
    seasonContentNode = m.seasonsList.content.GetChild(_seasonIndex)
    if seasonContentNode <> invalid
        regxp = CreateObject("roRegex", "\b     \b", "i")
        seasonTitle = regxp.Split(seasonContentNode.title)
        seasonContentNode.title = seasonTitle[0]
    end if
end function

'''''''''
' onOptionVoteButtonSelected: Handles vote button click on the Description Dialog.
' 
'''''''''
sub onOptionVoteButtonSelected()
    m.descriptionDialog.close = true
    if m.contentVoting = "true"
        episodes = m.episodesList.content.GetChildren(- 1, 0)
        selectedEpisode = episodes[m.episodesList.itemFocused]
        VoteForContent(selectedEpisode.id, m.descriptionDialog.buttons[m.descriptionDialog.buttonSelected], m.top)
    end if
end sub'//# sourceMappingURL=./SeasonsWithEpisodesScreen.bs.map